-- Fichero FSQL_SEM.sql
-------------------------------------------------------------------------------------
-- ***** ***** ***** ***** ***  ANALIZADOR  SEMNTICO  *** ***** ***** ***** ***** --
-- Encargado de traducir una consulta FSQL correcta lxica y sintcticamente en la --
-- consulta SQL equivalente. Resultado: Concatenando los FSQL_QUERY.atributo       --
-- Usa tablas: FSQL_ERRORS y FSQL_QUERY (Tablas que usa el cliente FSQL)           --
--             ACCESSIBLE_TABLES: Vista que contiene las tablas, vistas...         --
--                                que son accesibles por el USER actual.           --
--                                Atributos: OWNER, OBJ#, TABLE_NAME y TABLE_TYPE. --
--             DBA_TAB_COLUMNS: Vista con las columnas de todas las tablas         --
-------------------------------------------------------------------------------------
CREATE OR REPLACE PACKAGE FSQL_SEMANTIC AS
  FUNCTION semantico RETURN INTEGER;
END FSQL_SEMANTIC;
/
-- select * from all_source where name='FSQL_SEMANTIC' and line=2745;
CREATE OR REPLACE PACKAGE BODY FSQL_SEMANTIC AS
-- Tipo tabla PL/SQL para formar la expresin que calcula el Grado de Compatibilidad.
-- En la tabla se introduce la condicin del WHERE y posteriormente se resuelven
-- los operadores lgicos: NOT A (1-A), A AND B (min(A,B)) y A OR B (max(A,B)).
TYPE TABLA_CDEG IS TABLE OF FSQL_QUERY.atributo%TYPE INDEX BY BINARY_INTEGER;

-- Tipo FSQL_QUERY%ROWTYPE, quitndole el campo SESSIONID (que es constante)
TYPE QUERYTYPE IS RECORD (
  INDICE   FSQL_QUERY.INDICE%TYPE,
  POSICION FSQL_QUERY.POSICION%TYPE,
  NOMBRE   FSQL_QUERY.NOMBRE%TYPE,
  ATRIBUTO FSQL_QUERY.ATRIBUTO%TYPE);

-- Variable global con el nmero de Sesin actual (es constante en toda la ejecucin)
AUDSID FSQL_QUERY.SESSIONID%TYPE;

-- Registro para la tabla de alias:
TYPE REG_ALIAS IS RECORD (
     N_alias  VARCHAR2(30),  -- Nombre del alias de tabla puesto en clausula FROM
     N_scheme VARCHAR2(30),  -- Nombre del Esquema (owner, propietario) de la tabla
     N_table  VARCHAR2(30)); -- Nombre de la Tabla correspondiente a ese alias

-- Var. global ALL_ALIAS: Tabla con todos los alias usados:
TYPE TABLA_ALIAS IS TABLE OF REG_ALIAS NOT NULL INDEX BY BINARY_INTEGER;
ALL_ALIAS TABLA_ALIAS;

-- Vars. globales para las funciones (T-normas...) a aplicar en el CDEG
-- con operadores lgicos: NOT, AND y OR.
-- Son calculados en la funcin Lee_OPTIONS y son usados en
-- las funciones cambia_oper_logicos y cambia_oper_logicos2
F_NOT FSQL_OPTIONS.VALOR%TYPE;
F_AND FSQL_OPTIONS.VALOR%TYPE;
F_OR  FSQL_OPTIONS.VALOR%TYPE;
TRATA_FUZZY_ATRIB FSQL_OPTIONS.VALOR%TYPE;

PROCEDURE HALT IS -- Para detener la ejecucin en fase de pruebas
A INTEGER;
BEGIN
  COMMIT; SELECT INDICE INTO A FROM FSQL_QUERY WHERE ATRIBUTO='ALGO IMPOSIBLExx';
END HALT;

-------------------------------------------------------------------------------------
-- Devuelve el propietario y el nombre de una tabla real dada la tabla usada en la consulta:
-- Si el nombre usado es un alias devuelve ea propietario y tabla correspondiente y si no,
-- devuelve el owner y tabla propuestos originalmente (dados como argumentos).
-------------------------------------------------------------------------------------
PROCEDURE Nombre_Owner_Table (Sch  IN OUT VARCHAR2,
                              Tab  IN OUT VARCHAR2) IS
  i INTEGER;
BEGIN
  FOR i IN 1..ALL_ALIAS.COUNT LOOP
      IF ALL_ALIAS(i).N_alias=Tab THEN
         Sch:=ALL_ALIAS(i).N_scheme;
         Tab:=ALL_ALIAS(i).N_table;
         RETURN;
      END IF;
  END LOOP;
END Nombre_Owner_Table;

-------------------------------------------------------------------------------------
-- Devuelve el Identificador de Objeto (OBJ#) de una tabla real dado el nombre de la
-- tabla usada en la consulta: Si el nombre usado es un alias devuelve el OBJ# de la
-- tabla a la que corresponde y si no, devuelve el OBJ# de la tabla dada.
-------------------------------------------------------------------------------------
FUNCTION OBJ_ID (SchemeIN IN VARCHAR2,
                 TableIN  IN VARCHAR2) RETURN ACCESSIBLE_TABLES.OBJ#%TYPE IS
  i   INTEGER;
  OBJ ACCESSIBLE_TABLES.OBJ#%TYPE;
BEGIN
  -- Ver si es un alias
  FOR i IN 1..ALL_ALIAS.COUNT LOOP
      IF ALL_ALIAS(i).N_alias=TableIN THEN
         SELECT OBJ# INTO OBJ FROM ACCESSIBLE_TABLES
           WHERE OWNER=ALL_ALIAS(i).N_scheme AND TABLE_NAME=ALL_ALIAS(i).N_table;
         RETURN OBJ;
      END IF;
  END LOOP;
  -- Si NO existe un alias con ese nombre, devolvemos el OBJ# del schemeIN.TablaIN
  SELECT OBJ# INTO OBJ FROM ACCESSIBLE_TABLES
    WHERE OWNER=SchemeIN AND TABLE_NAME=TableIN;
  RETURN OBJ;
EXCEPTION WHEN NO_DATA_FOUND THEN -- Error: No tiene definidos los valores
  FSQL_AUX.inserta_error('ERROR SEMNTICO: Tabla, vista o alias mal situado ('
    ||SchemeIN||'.'||TableIN||').');
  RETURN 0;
END OBJ_ID;

-------------------------------------------------------------------------------------
-- Devuelve el Nombre de una Columna cuyos datos se dan por argumentos.
-- Tiene en cuenta si el nombre de tabla dado es un alias.
-------------------------------------------------------------------------------------
FUNCTION Nombre_Columna (SchemeIN IN VARCHAR2,
                         TableIN  IN VARCHAR2,
                         Col_ID   IN DBA_TAB_COLUMNS.COLUMN_ID%TYPE) RETURN VARCHAR2 IS
  i   INTEGER;
  Col DBA_TAB_COLUMNS.COLUMN_NAME%TYPE;
BEGIN
  -- Ver si es un alias
  FOR i IN 1..ALL_ALIAS.COUNT LOOP
      IF ALL_ALIAS(i).N_alias=TableIN THEN
         SELECT COLUMN_NAME INTO Col FROM DBA_TAB_COLUMNS
           WHERE OWNER=ALL_ALIAS(i).N_scheme AND TABLE_NAME=ALL_ALIAS(i).N_table AND
                 COLUMN_ID=Col_ID;
         RETURN Col;
      END IF;
  END LOOP;
  -- Si NO existe un alias con ese nombre:
  SELECT COLUMN_NAME INTO Col FROM DBA_TAB_COLUMNS
    WHERE OWNER=SchemeIN AND TABLE_NAME=TableIN AND COLUMN_ID=Col_ID;
  RETURN Col;
END Nombre_Columna;

-------------------------------------------------------------------------------------
-- Devuelve el Nmero de Columnas de una determinada tabla o alias de tabla.
-------------------------------------------------------------------------------------
FUNCTION Numero_Columnas (SchemeIN IN VARCHAR2,
                          TableIN  IN VARCHAR2) RETURN INTEGER IS
  i            INTEGER;
  num_columnas INTEGER;
BEGIN
  -- Ver si es un alias
  FOR i IN 1..ALL_ALIAS.COUNT LOOP
      IF ALL_ALIAS(i).N_alias=TableIN THEN
         SELECT COUNT(*) INTO num_columnas FROM DBA_TAB_COLUMNS
           WHERE OWNER     =ALL_ALIAS(i).N_scheme AND
                 TABLE_NAME=ALL_ALIAS(i).N_table;
         RETURN num_columnas;
      END IF;
  END LOOP;
  -- Si NO existe un alias con ese nombre:
  SELECT COUNT(*) INTO num_columnas FROM DBA_TAB_COLUMNS
    WHERE OWNER     =SchemeIN AND
          TABLE_NAME=TableIN;
  RETURN num_columnas;
END Numero_Columnas;

-------------------------------------------------------------------------------------
-- Devuelve el Nmero de Columnas de una determinada tabla o alias de tabla.
-------------------------------------------------------------------------------------
FUNCTION Hay_Columna_en_tabla (SchemeIN IN VARCHAR2,
                               TableIN  IN VARCHAR2,
                               ColIN    IN VARCHAR2) RETURN BOOLEAN IS
  i      INTEGER;
  n_objs INTEGER;
BEGIN
  -- Ver si es un alias
  FOR i IN 1..ALL_ALIAS.COUNT LOOP
      IF ALL_ALIAS(i).N_alias=TableIN THEN
         SELECT COUNT(*) INTO n_objs FROM DBA_TAB_COLUMNS
           WHERE TABLE_NAME =ALL_ALIAS(i).N_table  AND
                 OWNER      =ALL_ALIAS(i).N_scheme AND
                 COLUMN_NAME=ColIN;
         RETURN (n_objs=1); -- Devuelve TRUE si existe la columna
      END IF;
  END LOOP;
  -- Si NO existe un alias con ese nombre:
  SELECT COUNT(*) INTO n_objs FROM DBA_TAB_COLUMNS
    WHERE TABLE_NAME =TableIN  AND
          OWNER      =SchemeIN AND
          COLUMN_NAME=ColIN;
  RETURN (n_objs=1); -- Devuelve TRUE si existe la columna
END Hay_Columna_en_tabla;

----------------------------------------------------------------------------------------------
-- Devuelve el propietario de una tabla/vista cuyo nombre se pasa por argumentos (tab)
-- Si se existe algn error, devuelve NULL;
-- Naturalmente, esta funcin se usa cuando el usuario no especifica el propietario en la consulta.
-- Esta funcin comprueba que esa tabla est puesta en la clusula FROM
--    a) Si no aparece en la clusula FROM genera un error
--  b) Si hay una tabla con ese nombre en el FROM, se devuelve su propietario
--  c) Si hay ms de una tabla con ese nombre en el FROM:
--       Seguramente pertenecen a distintos FROM (de subconsultas).
-- Requisitos: Se supone que YA se ha chequeado que las tablas del FROM existen y son
--             accesibles por el usuario actual.
----------------------------------------------------------------------------------------------
FUNCTION Propietario_Tabla (tab IN QUERYTYPE) RETURN VARCHAR2 IS
  n_objs      INTEGER;
  Propietario VARCHAR2(30);
  Fila        QUERYTYPE;
  -- Cursor para las tablas de la consulta (clusula FROM):
  CURSOR tablas IS
     SELECT indice,posicion,nombre,atributo FROM FSQL_QUERY
     WHERE  nombre LIKE '%tabla';
BEGIN
  -- Ver si es un alias: Si lo es, ya tenemos su propietario
  FOR i IN 1..ALL_ALIAS.COUNT LOOP
      IF ALL_ALIAS(i).N_alias=tab.atributo THEN
         RETURN ALL_ALIAS(i).N_scheme;
      END IF;
  END LOOP;

  -- Contar el nmero de tablas del FROM con ese nombre
  SELECT count(*) INTO n_objs FROM FSQL_QUERY
    WHERE nombre LIKE '%tabla' AND atributo=tab.atributo;

  IF n_objs=0 THEN -- Caso a)
--   Podramos no dar aqui el error: Este lo dara Oracle al mandar la consulta
     FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||tab.posicion
       ||': Tabla o vista '||tab.atributo||' no consta en la clusula FROM.');
     RETURN NULL;
  ELSIF n_objs=1 THEN -- Caso b)
     SELECT indice,posicion,nombre,atributo INTO Fila FROM FSQL_QUERY
       WHERE nombre LIKE '%tabla' AND atributo=tab.atributo;
     IF Fila.nombre='tabla' THEN
        -- Si no se pone el propietario se supone el usuario
        SELECT USER into Propietario from DUAL;
     ELSE -- Fila.nombre='s.tabla' THEN
        SELECT atributo INTO Propietario FROM FSQL_QUERY
          WHERE indice=Fila.indice-2;
     END IF;
  ELSE -- Caso c)
    SELECT count(*) INTO n_objs FROM FSQL_QUERY
      WHERE nombre='tabla' AND atributo=tab.atributo;
    IF n_objs<>0 then
       SELECT USER into Propietario from DUAL;
    ELSE
     SELECT indice,posicion,nombre,atributo INTO Fila FROM FSQL_QUERY
       WHERE nombre='s.tabla' AND atributo=tab.atributo AND
             indice=(SELECT MIN(indice) FROM FSQL_QUERY
                       WHERE nombre='s.tabla' AND atributo=tab.atributo);
     SELECT atributo INTO Propietario FROM FSQL_QUERY
       WHERE indice=Fila.indice-2;
    END IF;
  END IF;
  RETURN Propietario;
END Propietario_Tabla;

----------------------------------------------------------------------------------------------
-- Indica si existe o no la columna indicada en una tupla de FSQL_QUERY;
-- En caso afirmativo se devuelve tabla y esquema correspondientes schema.tab.columna
-- Si en la consulta no se facilita la tabla se buscar entre las tablas del FROM
----------------------------------------------------------------------------------------------
FUNCTION existe_columna (columna IN QUERYTYPE,
                         scheme  IN OUT VARCHAR2,
                         tab     IN OUT VARCHAR2) RETURN BOOLEAN
IS
  -- Cursor para las tablas de la consulta (clusula FROM):
  CURSOR tablas IS
     SELECT indice,posicion,nombre,atributo FROM FSQL_QUERY
     WHERE  nombre LIKE '%tabla';
  tabla  QUERYTYPE;        -- Tablas de la lista de tokens
  EncontradaTabla BOOLEAN; -- Si se ha encontrado la tabla de una columna
BEGIN
  IF columna.nombre='column' THEN
     -- Si no me indican la tabla donde est la columna hay que buscarla:
     EncontradaTabla:=FALSE;
     OPEN tablas;
     LOOP -- Salida del bucle: Si no hay ms tablas o encontramos la columna en una tabla
       FETCH tablas INTO tabla;
       EXIT WHEN tablas%NOTFOUND;
       IF tabla.nombre='tabla' THEN
          scheme:=Propietario_Tabla(tabla);
          IF scheme IS NULL THEN RETURN FALSE; END IF; -- La consulta contiene un error en esa tabla
       ELSE -- tabla.nombre='s.tabla' THEN
          SELECT atributo INTO scheme FROM FSQL_QUERY
            WHERE indice=tabla.indice-2;
       END IF;
--       SELECT COUNT(*) INTO n_objs FROM DBA_TAB_COLUMNS
--         WHERE TABLE_NAME =Nombre_Table(tabla.atributo) AND
--               OWNER      =Nombre_Owner(scheme,tabla.atributo) AND
--               COLUMN_NAME=columna.atributo;
       IF Hay_Columna_en_tabla(scheme,tabla.atributo,columna.atributo) THEN
          EncontradaTabla:=TRUE;
          tab:=tabla.atributo; -- scheme.tab
          EXIT;
       END IF;
     END LOOP;
     CLOSE tablas;

     IF EncontradaTabla THEN
        RETURN TRUE;
     ELSE
        RETURN FALSE;
     END IF;

  -- Si me indican la tabla de la columna:
  ELSIF columna.nombre='t.column' THEN
     SELECT indice,posicion,nombre,atributo INTO tabla FROM FSQL_QUERY
       WHERE indice=columna.indice-2;
     tab:=tabla.atributo;
     scheme:=Propietario_Tabla(tabla);
     IF scheme IS NULL THEN RETURN FALSE; END IF; -- La consulta contiene un error en esa tabla
  ELSE --columna.nombre='s.t.column'
     SELECT atributo INTO tab FROM FSQL_QUERY
       WHERE indice=columna.indice-2;
     SELECT atributo INTO scheme FROM FSQL_QUERY
       WHERE indice=columna.indice-4;
  END IF;
--  SELECT COUNT(*) INTO n_objs FROM DBA_TAB_COLUMNS
--    WHERE OWNER     =Nombre_Owner(scheme,tab) AND
--          TABLE_NAME=Nombre_Table(tab)        AND
--          COLUMN_NAME=columna.atributo;
  RETURN Hay_Columna_en_tabla(scheme,tab,columna.atributo);
  -- Si NO existe se devuelve FALSE y si existe se devuelve TRUE.
  -- En este ltimo caso habra que comprobar que la tabla aparece en la clusula
  -- FROM: Este error lo generar Oracle: "ORA-00904: invalid column name"
END existe_columna;

----------------------------------------------------------------------------------------------
-- Devuelve el tipo fuzzy (1,2  3) de la columna en la tabla FUZZY_COL_LIST (FCL)
-- Si existe en FCL, adems, devuelve en OBJ y COL los identificadores de la columna
-- Si no existe la columna en FCL, devuelve 0.
----------------------------------------------------------------------------------------------
FUNCTION tipo_en_FCL(scheme  IN VARCHAR2,
                     tabla   IN VARCHAR2,
                     columna IN VARCHAR2,
                     OBJ IN OUT FUZZY_COL_LIST.OBJ#%TYPE,
                     COL IN OUT FUZZY_COL_LIST.COL#%TYPE) RETURN FUZZY_COL_LIST.F_TYPE%TYPE
IS
  tipo FUZZY_COL_LIST.F_TYPE%TYPE;
  s VARCHAR2(30):=scheme;
  t VARCHAR2(30):=tabla;
BEGIN
  OBJ:=OBJ_ID(scheme,tabla);
  Nombre_Owner_Table(s,t); -- Hallamos los nombres reales (por si es un alias)
  SELECT COLUMN_ID INTO COL FROM DBA_TAB_COLUMNS
    WHERE OWNER=s AND TABLE_NAME=t AND COLUMN_NAME=columna;
  SELECT F_TYPE INTO tipo FROM FUZZY_COL_LIST
    WHERE OBJ#=OBJ AND COL#=COL;
  RETURN tipo;
EXCEPTION
  WHEN NO_DATA_FOUND THEN RETURN 0;
END tipo_en_FCL;

----------------------------------------------------------------------------------------------
-- Examina una columna de la consulta (FSQL_QUERY) para ver si existe o no.
-- Si no existe (como crisp o como difusa) inserta un error en FSQL_ERRORS
-- Si existe, devuelve si una columna de la consulta es difusa o no
-- Obtiene sus valores de OBJ (Identificador de tabla), COL y su tipo difuso
----------------------------------------------------------------------------------------------
FUNCTION es_fuzzy_column (columna IN QUERYTYPE,
                          scheme  IN OUT VARCHAR2,
                          tab     IN OUT VARCHAR2,
                          f_tipo  IN OUT FUZZY_COL_LIST.F_TYPE%TYPE,
                          OBJ IN OUT FUZZY_COL_LIST.OBJ#%TYPE,
                          COL IN OUT FUZZY_COL_LIST.COL#%TYPE) RETURN BOOLEAN
IS
  column QUERYTYPE;
BEGIN
  column:=columna;
--dbms_output.put_line('*****:'||column.nombre||':'||column.atributo);
  IF column.nombre='*column' THEN -- Es un * dentro de CDEG
     RETURN TRUE; -- Se tratar ms adelante
  ELSIF column.atributo IN ('CURRVAL','NEXTVAL','LEVEL','ROWID','ROWNUM') THEN
     UPDATE FSQL_QUERY set nombre='crisp_col' WHERE indice=column.indice;
     RETURN FALSE;
  ELSIF existe_columna(column,scheme,tab) THEN
     -- Esa columna es crisp: Ver si admite tratamiento difuso
     f_tipo:=tipo_en_FCL(scheme,tab,column.atributo,OBJ,COL);
     IF f_tipo<>1 THEN -- Columna crisp, sin tratamiento difuso: La marcamos como tal
        UPDATE FSQL_QUERY set nombre='crisp_col'
          WHERE indice=column.indice;
        RETURN FALSE;
     END IF;
     RETURN TRUE; -- f_tipo=1: Columna crisp con tratamiento difuso
  ELSE
     -- Si no existe la columna como crisp, veamos si existe como difusa:
     column.atributo:=column.atributo||'T';
     IF NOT existe_columna(column,scheme,tab) THEN -- Error: Columna no vlida
        -- La ponemos como tratada para que no vuelva a producir el mismo error
--dbms_output.put_line(':'||column.indice||':'||column.nombre);
--dbms_output.put_line(':'||column.atributo||':'||scheme||':'||tab);
        UPDATE FSQL_QUERY set nombre='ok' WHERE indice=column.indice;
        IF column.nombre='column' THEN
           FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||column.posicion
             ||': Nombre de columna '||substr(column.atributo,1,length(column.atributo)-1)
             ||' no vlido.');
        ELSIF column.nombre='t.column' THEN
           FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||column.posicion
             ||': Nombre de columna '||tab||'.'||substr(column.atributo,1,length(column.atributo)-1)
             ||' no vlido.');
        ELSE
           FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||column.posicion
             ||': Nombre de columna '||scheme||'.'||tab||'.'
             ||substr(column.atributo,1,length(column.atributo)-1)
             ||' no vlido.');
        END IF;
     ELSE -- Existe la columna como DIFUSA: Ver si existe en FCL y tratarla como tal
        f_tipo:=tipo_en_FCL(scheme,tab,column.atributo,OBJ,COL);
        column.atributo:=substr(column.atributo,1,length(column.atributo)-1);--Quito la 'T'
        IF f_tipo=0 THEN -- Columna no existe en FCL
           FSQL_AUX.inserta_error('ERROR en FIRST en posicin '||column.posicion||
             ': Columna difusa '||scheme||'.'||tab||'.'||column.atributo||
             ' no existe en FUZZY_COL_LIST, (OBJ#,COL#)=('||OBJ||','||COL||').');
        ELSE
          RETURN TRUE;
        END IF;
     END IF;
  END IF;
  RETURN FALSE;
END es_fuzzy_column;

----------------------------------------------------------------------------------------------
-- Realiza el tratamiento para las columnas difusas
-- Modifica el atributo de FSQL_QUERY
-- Una columna difusa puede aparecer en los siguientes lugares:
--  1. En la select_list como columna solitaria (no en expresiones)
--  2. Como argumento de la funcin CDEG(fcolumn): Estas columnas se marcarn en el campo
--     nombre con: scheme||'.'||tab||'.'||atributo||'\CDEG'
--     (Si es un CDEG(*) se pondr '*\CDEG' en el campo nombre)
--     Esta funcin se admite en cualquier lugar (lo habitual ser en la select_list)
--  3. Justo antes de un comparador difuso (FEQ, FGT...)
--    4. A la derecha de un comparador difuso si hay otra columna a la izquierda,
--       del mismo tipo (2  3). En esta posicin, un atributo tipo 1 no acta como difuso.
--    5. Antes de un: IS [NOT] {UNKNOWN|UNDEFINED|NULL]
-- En cualquier otro lugar ocasionar un Error Semntico (Atributo difuso mal situado)
--       o es tratado como su representacin grfica (funciones fshow2 y fshow3).
-- Las columnas tratadas se marcan en el campo nombre de FSQL_QUERY con:
--  scheme||'.'||tab||'.'||columna.atributo||'\'   Si la columna est en una condicin
--                   "                       '\2\' Si es la columna 2 de una comparacion
--  'ok'                                           en otro caso
-- Esta forma de marcar columnas tratadas nos ayudar al calcular CDEG (proc. calcula_CDEG)
-- La '\' es para evitar confusiones (si alguna columna se llamara 'column')
----------------------------------------------------------------------------------------------
PROCEDURE trata_fuzzy_column (columna IN QUERYTYPE,
                              scheme  IN OUT VARCHAR2,
                              tab     IN OUT VARCHAR2,
                              f_tipo  IN FUZZY_COL_LIST.F_TYPE%TYPE,
                              OBJ IN OUT FUZZY_COL_LIST.OBJ#%TYPE,
                              COL IN OUT FUZZY_COL_LIST.COL#%TYPE)
IS
  token_ant FSQL_QUERY.nombre%TYPE; -- Token anterior a la especificacin de la columna
  token_sig FSQL_QUERY.nombre%TYPE; -- Token siguiente
  alias     VARCHAR2(30); -- Alias de una columna difusa, si no tiene en la consulta
  st        VARCHAR2(62); -- Scheme.tabla a concatenar con el nombre de columna (si procede)
  margen    FUZZY_APPROX_MUCH.MARGEN%TYPE; -- Si encontramos un APROX, hallar su margen
  much      FUZZY_APPROX_MUCH.MUCH%TYPE;   -- Si se aplican los comparadores MGT, MLT, NMGT o NMLT
  umbral    VARCHAR2(2);  -- Clculo de la cadena del umbral a aadir tras la llamada
  fcomp     VARCHAR2(5);  -- Fuzzy comparador
  fcompSinN VARCHAR2(4);  -- Fuzzy comparador, sin 'N' (donde corresponda)
  alfa      FSQL_QUERY.atributo%TYPE; -- Parmetros de un trapecio
  beta      FSQL_QUERY.atributo%TYPE;
  gamma     FSQL_QUERY.atributo%TYPE;
  delta     FSQL_QUERY.atributo%TYPE;
  F_ID      FUZZY_LABEL_DEF.FUZZY_ID%TYPE; -- Ident. de una etiqueta
  pos       FSQL_QUERY.posicion%TYPE; --Calculo de la posicion de una etiqueta caso de error
  columna2  QUERYTYPE; -- Segunda columna difusa en una comparacin
  scheme2   FSQL_QUERY.nombre%TYPE;
  tab2      FSQL_QUERY.nombre%TYPE;
  f_tipo2   FUZZY_COL_LIST.F_TYPE%TYPE;
  OBJ2      FUZZY_COL_LIST.OBJ#%TYPE;
  COL2      FUZZY_COL_LIST.COL#%TYPE;
  ind       FSQL_QUERY.indice%TYPE;  -- Para moverse por el indice de la tabla
  length1   FUZZY_COL_LIST.LEN%TYPE; -- Nmero de etiquetas mximo en atributos tipo 3
  length2   FUZZY_COL_LIST.LEN%TYPE; -- Nmero de etiquetas mximo en atributos tipo 3
  argstipo3 VARCHAR2(1500);          -- Para formar la llamada a la funcin FEQ3_label y FEQ3

BEGIN
  IF columna.nombre='*column' THEN -- Se trata de un CDEG(*)
     UPDATE FSQL_QUERY SET nombre='*\CDEG' WHERE indice=columna.indice;
     -- Borro el 'CDEG('
     UPDATE FSQL_QUERY SET atributo=NULL
       WHERE indice BETWEEN columna.indice-2 AND columna.indice-1;
     RETURN;
  END IF;

  -- Calculamos el token anterior a la especificacin de columna (s.t.c) y
  -- borramos con un NULL el scheme (s), la tabla (t) y los puntos de separacin, si existen:
  -- dbms_output.put_line(columna.posicion);
  IF columna.nombre='column' THEN
    st:=NULL;
    SELECT nombre,posicion INTO token_ant,pos FROM FSQL_QUERY
      WHERE indice=columna.indice-1;
  ELSIF columna.nombre='t.column' THEN
    st:=tab||'.';
    SELECT nombre,posicion INTO token_ant,pos FROM FSQL_QUERY
      WHERE indice=columna.indice-3;
    UPDATE FSQL_QUERY SET atributo=NULL
      WHERE indice BETWEEN columna.indice-2 AND columna.indice-1;
  ELSIF columna.nombre='s.t.column' THEN -- caso scheme.tabla.columna
    st:=scheme||'.'||tab||'.';
    SELECT nombre,posicion INTO token_ant,pos FROM FSQL_QUERY
      WHERE indice=columna.indice-5;
    UPDATE FSQL_QUERY SET atributo=NULL
      WHERE indice BETWEEN columna.indice-4 AND columna.indice-1;
  END IF;

  -- Calculamos el token siguiente a la especificacin de columna
  BEGIN
    SELECT nombre INTO token_sig FROM FSQL_QUERY
      WHERE indice=columna.indice+1;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN token_sig:=NULL; -- Columna al final de la consulta
  END;

--***** ***** ***** ***** ***** SELECT LIST ***** ***** ***** ***** *****
  -- Para ver si estamos en la select_list miramos los tokens que rodean la columna
  IF (token_ant IN ('SELECT',', en sl','ALL','DISTINCT')) AND
     (token_sig IN ('FROM','c_alias',', en sl')) THEN
     -- Nos encontramos en la select_list, con un atributo independiente:
     -- Lo cambiamos por la funcin para mostrar atributos difusos, si es tipo 2  3.
     -- Si es tipo 1 no hay que cambiar nada en la select_list.
     IF f_tipo=2 OR f_tipo=3 THEN
        IF token_sig<>'c_alias' THEN -- Si no tiene alias, se lo ponemos
           alias:=' '||columna.atributo;
        ELSE alias:=NULL;
        END IF;
        FSQL_AUX.fuzzy_column_en_sl(st||columna.atributo,columna.indice,f_tipo,OBJ,COL,alias);
     END IF;

  ELSIF token_sig=') function' THEN   -- Ver si es un CDEG(fcolumn)
    -- Calculamos el token anterior al supuesto '( function',
    -- que son los parntesis de las funciones:
    IF columna.nombre='column' THEN
       ind:=columna.indice-2;
       SELECT nombre INTO token_ant FROM FSQL_QUERY WHERE indice=ind;
    ELSIF columna.nombre='t.column' THEN
       ind:=columna.indice-4;
       SELECT nombre INTO token_ant FROM FSQL_QUERY WHERE indice=ind;
    ELSE -- caso scheme.tabla.columna
       ind:=columna.indice-6;
       SELECT nombre INTO token_ant FROM FSQL_QUERY WHERE indice=ind;
    END IF;
    IF token_ant='CDEG' THEN -- Tenemos un CDEG(fcolumn) --> tipos 1, 2  3 <--
       -- Marco esta columna para su tratamiento posterior
       UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||atributo||'\CDEG'
         WHERE indice=columna.indice;
       -- Borro el 'CDEG('
       UPDATE FSQL_QUERY SET atributo=NULL
         WHERE indice BETWEEN ind AND ind+1;
    ELSE
       -- Estamos en una funcin que NO es CDEG: Calcular la llamada a la
       -- funcin pertinente, igual que si estuviera en la select_list
--dbms_output.put_line(f_tipo||'-'||columna.atributo||'-'||columna.indice);
       IF f_tipo=2 OR f_tipo=3 THEN
          FSQL_AUX.fuzzy_column_en_sl(st||columna.atributo,columna.indice,f_tipo,OBJ,COL,alias);
       END IF;
    END IF;

-- ***** ***** Clusula WHERE: 8 Tipos de condiciones fuzzy ***** *****
--  fcol <fcomp> #n         fcol <fcomp> $[a,b,c,d]
--  fcol <fcomp> [n,m]      fcol <fcomp> $label
--  fcol <fcomp> fcol2      fcol <fcomp> expr_crisp
--  fcol <fcomp>  {UNKNOWN|UNDEFINED|NULL}
--  fcol IS [NOT] {UNKNOWN|UNDEFINED|NULL}

--***** ***** ***** Antes de un fcomparador: fcol <fcomp> <expr> ***** ***** *****
  ELSIF token_sig IN ('FGT','FLT','FEQ','FGEQ','FLEQ','MGT','MLT',
                      'NFGT','NFLT','NFEQ','NFGEQ','NFLEQ','NMGT','NMLT') THEN
    fcomp:=token_sig;
    UPDATE FSQL_QUERY SET atributo=NULL -- Borramos el fcomparador
      WHERE indice=columna.indice+1;
    -- Examinar operando tras el fcomparador:
    --  * Delimitado por THOLD, umbral(NUMERO), comparador, 'sin umbral' o FIN
    -- Observe que es legal ...WHERE .5*fcol FEQ #3 THOLD .2 -> ...WHERE .5*FEQ(fcol,#3)>=.2
    -- Eso no es operacin aritmtica sobre una columna difusa sino sobre una comparacin
    SELECT nombre INTO token_sig FROM FSQL_QUERY
      WHERE indice=columna.indice+2;

--***** ***** ***** VALOR APROXIMADO: fcol <fcomp> #n ***** ***** *****
    IF token_sig='#' THEN -- Tenemos un aprox
       IF f_tipo=3 THEN -- Error: No es comparable un tipo 3 con un aprox
          FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
            ': Columna difusa '||st||columna.atributo||
            ' es tipo 3 y no se puede comparar con un valor aproximado #n.');
       ELSE -- Tipos 1  2
         BEGIN
           -- Calcular margen (Si no existe se levanta una excepcin)
           SELECT MARGEN,MUCH INTO margen,much FROM FUZZY_APPROX_MUCH WHERE OBJ#=OBJ AND COL#=COL;
           -- Calcular nmero aprox (seguro que existe si la sentencia es correcta sintcticamente)
           SELECT atributo INTO token_sig FROM FSQL_QUERY
             WHERE indice=columna.indice+3;
           -- Sustituir columna por llamada a fcomp_aprox
           umbral:=FSQL_AUX.calcula_umbral(columna.indice+4);
           IF f_tipo=2 THEN
              IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN
                 -- Si son estos comparadores, calculamos y pasamos el margen y el much
                 FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
                 IF margen IS NULL THEN RETURN; END IF;
                 UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                   atributo='FSQL_FUNCTIONS2.'||fcomp||'_aprox('||
                   token_sig||','||TO_CHAR(TO_NUMBER(token_sig)-margen)||','||
                   TO_CHAR(TO_NUMBER(token_sig)+margen)||','||margen||','||
                   OBJ||','||COL||','||st||columna.atributo||'T,'||
                   st||columna.atributo||'1,'||st||columna.atributo||'2,'||
                   st||columna.atributo||'3,'||st||columna.atributo||'4,'
                   ||TO_CHAR(margen)||','||TO_CHAR(much)||')'||umbral
                   WHERE indice=columna.indice;
              ELSE
                 UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                   atributo='FSQL_FUNCTIONS.'||fcomp||'_aprox('||
                   token_sig||','||TO_CHAR(TO_NUMBER(token_sig)-margen)||','||
                   TO_CHAR(TO_NUMBER(token_sig)+margen)||','||margen||','||
                   OBJ||','||COL||','||st||columna.atributo||'T,'||
                   st||columna.atributo||'1,'||st||columna.atributo||'2,'||
                   st||columna.atributo||'3,'||st||columna.atributo||'4)'||umbral
                   WHERE indice=columna.indice;
              END IF;
           ELSE -- f_tipo=1
             IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN
                -- Si son estos comparadores, pasamos el margen y el much
                -- como argumentos (los cuales ya tenemos calculados):
                UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                  atributo='FSQL_FUNCTIONS2.'||fcomp||'_crisp_aprox('
                  ||st||columna.atributo||','||token_sig||','
                  ||TO_CHAR(TO_NUMBER(token_sig)-margen)||','
                  ||TO_CHAR(TO_NUMBER(token_sig)+margen)||','||margen
                  ||','||TO_CHAR(margen)||','||TO_CHAR(much)||')'||umbral
                  WHERE indice=columna.indice;
             ELSE
                fcompSinN:=FSQL_AUX.Sin_N(fcomp); -- Quitamos la N del comparador si es necesario
                UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                  atributo='FSQL_FUNCTIONS.'||fcompSinN||'_crisp_aprox('
                  ||st||columna.atributo||','||token_sig||','
                  ||TO_CHAR(TO_NUMBER(token_sig)-margen)||','
                  ||TO_CHAR(TO_NUMBER(token_sig)+margen)||','||margen||')'||umbral
                  WHERE indice=columna.indice;
             END IF;
           END IF;
           -- Borrar el nmero aprox: #n
           UPDATE FSQL_QUERY SET atributo=NULL
             WHERE indice BETWEEN columna.indice+2 AND columna.indice+3;
         EXCEPTION
           WHEN NO_DATA_FOUND THEN
             FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
               ': Columna difusa '||st||columna.atributo||', (OBJ#,COL#)=('||OBJ||','||COL||
               '), no tiene el margen establecido para poder usar valores aproximados #n:'||
               ' Introducir uno en la tabla FUZZY_APPROX_MUCH.');
         END;
       END IF;

--***** ***** ***** VALOR INTERVALO: fcol <fcomp> [n,m] ***** ***** *****
    ELSIF token_sig='[' THEN -- Intervalo
       IF f_tipo=3 THEN -- Error: No es comparable un tipo 3 con un intervalo
          FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
            ': Columna difusa '||st||columna.atributo||
            ' es tipo 3 y no se puede comparar con un intervalo [n,m].');
       ELSIF f_tipo=2 THEN
         IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN
            -- Sustituir columna por el principio de la llamada a fcomp_inter
            UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
              atributo='FSQL_FUNCTIONS2.'||fcomp||'_inter('
              WHERE indice=columna.indice;
            -- Borramos el '['
            UPDATE FSQL_QUERY SET atributo=NULL WHERE indice=columna.indice+2;
            -- Sustituimos el ']' por el resto de la llamada a fcomp_inter
            umbral:=FSQL_AUX.calcula_umbral(columna.indice+7);
            -- Si son estos comparadores, calculamos y pasamos el margen y el much
            FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
            IF margen IS NULL THEN RETURN; END IF;
            UPDATE FSQL_QUERY SET atributo=','||OBJ||','||COL||','
              ||st||columna.atributo||'T,'
              ||st||columna.atributo||'1,'||st||columna.atributo||'2,'
              ||st||columna.atributo||'3,'||st||columna.atributo||'4,'
              ||TO_CHAR(margen)||','||TO_CHAR(much)||')'||umbral
              WHERE indice=columna.indice+6;
         ELSE
            -- Sustituir columna por el principio de la llamada a fcomp_inter
            UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
              atributo='FSQL_FUNCTIONS.'||fcomp||'_inter('
              WHERE indice=columna.indice;
            -- Borramos el '['
            UPDATE FSQL_QUERY SET atributo=NULL WHERE indice=columna.indice+2;
            -- Sustituimos el ']' por el resto de la llamada a fcomp_inter
            umbral:=FSQL_AUX.calcula_umbral(columna.indice+7);
            UPDATE FSQL_QUERY SET atributo=','||OBJ||','||COL||','||
              st||columna.atributo||'T,'||
              st||columna.atributo||'1,'||st||columna.atributo||'2,'||
              st||columna.atributo||'3,'||st||columna.atributo||'4)'||umbral
              WHERE indice=columna.indice+6;
         END IF;
       ELSE -- f_tipo=1
         -- Comparacin del tipo: fcol_t1 FEQ [3,6]
         -- En ese caso, difuminamos el intervalo con el margen de fcol_t1
         -- Si no se desea difuminar usar comparacin crisp: fcol_t1 BETWEEN 3 AND 6
         -- Sustituir columna por el principio de la llamada a fcomp_crisp_finter
         fcompSinN:=FSQL_AUX.Sin_N(fcomp); -- Quitamos la N del comparador si es necesario
         IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN
            UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
              atributo='FSQL_FUNCTIONS2.'||fcompSinN||'_crisp_finter('||st||columna.atributo||','
              WHERE indice=columna.indice;
            -- Borramos el '['
            UPDATE FSQL_QUERY SET atributo=NULL WHERE indice=columna.indice+2;
            -- Sustituimos el ']' por el resto de la llamada a fcomp_crisp_finter
            umbral:=FSQL_AUX.calcula_umbral(columna.indice+7);
            -- Calculamos el margen y el much (para pasarlos por argumentos, si procede):
            FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
            IF margen IS NULL THEN RETURN; END IF;
            -- Si son estos comparadores pasamos el margen y el much como argumentos:
            UPDATE FSQL_QUERY SET atributo=','||TO_CHAR(margen)||','||TO_CHAR(much)||')'||umbral
              WHERE indice=columna.indice+6;
         ELSE
            UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
              atributo='FSQL_FUNCTIONS.'||fcompSinN||'_crisp_finter('||st||columna.atributo||','
              WHERE indice=columna.indice;
            -- Borramos el '['
            UPDATE FSQL_QUERY SET atributo=NULL WHERE indice=columna.indice+2;
            -- Sustituimos el ']' por el resto de la llamada a fcomp_crisp_finter
            umbral:=FSQL_AUX.calcula_umbral(columna.indice+7);
            -- Calculamos el margen y el much (para pasarlos por argumentos, si procede):
            FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
            IF margen IS NULL THEN RETURN; END IF;
            UPDATE FSQL_QUERY SET atributo=','||TO_CHAR(margen)||')'||umbral
              WHERE indice=columna.indice+6;
         END IF;
       END IF;

    ELSIF token_sig='$' THEN -- Trapecio o Label
      SELECT nombre INTO token_sig FROM FSQL_QUERY
        WHERE indice=columna.indice+3;

--***** ***** ***** VALOR TRAPECIO: fcol <fcomp> $[a,b,c,d] ***** ***** *****
      IF token_sig='[' THEN -- Trapecio
        IF f_tipo=3 THEN -- Error: No es comparable un tipo 3 con un trapecio
           FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
             ': Columna difusa '||st||columna.atributo||
             ' es tipo 3 y no se puede comparar con un trapecio $[a,b,c,d].');
        ELSE
          -- Hallar trapecio, ya que la llamada ser:  alga,beta-alfa,gamma-delta,delta
          SELECT atributo INTO alfa  FROM FSQL_QUERY WHERE indice=columna.indice+4;
          SELECT atributo INTO beta  FROM FSQL_QUERY WHERE indice=columna.indice+6;
          SELECT atributo INTO gamma FROM FSQL_QUERY WHERE indice=columna.indice+8;
          SELECT atributo INTO delta FROM FSQL_QUERY WHERE indice=columna.indice+10;
          -- Comprobar que el trapecio est bien definido:
          IF TO_NUMBER(alfa) >TO_NUMBER(beta)  OR
             TO_NUMBER(beta) >TO_NUMBER(gamma) OR
             TO_NUMBER(gamma)>TO_NUMBER(delta) THEN
             FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion
             ||': Trapecio posibilstico mal definido: $[a,b,c,d] tal que a<=b<=c<=d.'
             ||' Valores actuales: $['||TO_NUMBER(alfa)||','||TO_NUMBER(beta)||','
             ||TO_NUMBER(gamma)||','||TO_NUMBER(delta)||'].');
          END IF;
          umbral:=FSQL_AUX.calcula_umbral(columna.indice+12);
          IF f_tipo=2 THEN
             -- Sustituir columna por llamada a fcomp_trape
             IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN
                -- Si son estos comparadores, calculamos el margen y el much
                FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
                IF margen IS NULL THEN RETURN; END IF;
                UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                  atributo='FSQL_FUNCTIONS2.'||fcomp||'_trape('
                  ||alfa||','||TO_CHAR(TO_NUMBER(beta)-TO_NUMBER(alfa))||','
                  ||TO_CHAR(TO_NUMBER(gamma)-TO_NUMBER(delta))||','||delta||','
                  ||OBJ||','||COL||','||st||columna.atributo||'T,'||
                  st||columna.atributo||'1,'||st||columna.atributo||'2,'||
                  st||columna.atributo||'3,'||st||columna.atributo||'4,'
                  ||TO_CHAR(margen)||','||TO_CHAR(much)||')'||umbral
                  WHERE indice=columna.indice;
             ELSE
                UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                  atributo='FSQL_FUNCTIONS.'||fcomp||'_trape('
                  ||alfa||','||TO_CHAR(TO_NUMBER(beta)-TO_NUMBER(alfa))||','
                  ||TO_CHAR(TO_NUMBER(gamma)-TO_NUMBER(delta))||','||delta||','
                  ||OBJ||','||COL||','||st||columna.atributo||'T,'||
                  st||columna.atributo||'1,'||st||columna.atributo||'2,'||
                  st||columna.atributo||'3,'||st||columna.atributo||'4)'||umbral
                  WHERE indice=columna.indice;
             END IF;

          ELSE -- f_tipo=1
             -- Sustituir columna por llamada a fcomp_crisp_trape
             IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN
                -- Si son estos comparadores, calculamos el margen y el much
                FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
                IF margen IS NULL THEN RETURN; END IF;
                UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                  atributo='FSQL_FUNCTIONS2.'||fcomp||'_crisp_trape('||st||columna.atributo||','
                  ||alfa||','||TO_CHAR(TO_NUMBER(beta)-TO_NUMBER(alfa))||','
                  ||TO_CHAR(TO_NUMBER(gamma)-TO_NUMBER(delta))||','||delta
                  ||','||TO_CHAR(margen)||','||TO_CHAR(much)||')'||umbral
                  WHERE indice=columna.indice;
             ELSE
                fcompSinN:=FSQL_AUX.Sin_N(fcomp); -- Quitamos la N del comparador si es necesario
                UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                  atributo='FSQL_FUNCTIONS.'||fcompSinN||'_crisp_trape('||st||columna.atributo||','
                  ||alfa||','||TO_CHAR(TO_NUMBER(beta)-TO_NUMBER(alfa))||','
                  ||TO_CHAR(TO_NUMBER(gamma)-TO_NUMBER(delta))||','||delta||')'||umbral
                  WHERE indice=columna.indice;
             END IF;
          END IF;
          -- Borramos el desde el '$' hasta el ']'
          UPDATE FSQL_QUERY SET atributo=NULL
            WHERE indice BETWEEN columna.indice+2 AND columna.indice+11;
        END IF;

--***** ***** ***** VALOR ETIQUETA: fcol <fcomp> $label ***** ***** *****
      ELSE -- Label: Hallar etiqueta (alfa):
        SELECT atributo,posicion INTO alfa,pos FROM FSQL_QUERY
          WHERE indice=columna.indice+3;
        umbral:=FSQL_AUX.calcula_umbral(columna.indice+4);
        IF f_tipo=2 THEN
          BEGIN
            --  Hallar su identificador (F_ID) en FOL (Si no existe, salta una EXCEPCION)
            SELECT FUZZY_ID INTO F_ID FROM FUZZY_OBJECT_LIST
              WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_NAME=alfa;
            -- Calcular trapecio asociado a esa etiqueta en FLD
            -- (Obviamente, en este caso se supone que la etiqueta est definida sobre la
            --  misma columna con la que se esta comparando)
            BEGIN
              SELECT ALFA,BETA,GAMMA,DELTA INTO alfa,beta,gamma,delta FROM FUZZY_LABEL_DEF
                WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F_ID;
              -- Comprobar que el trapecio de la etiqueta est bien definido:
              IF TO_NUMBER(alfa) >TO_NUMBER(beta)  OR
                 TO_NUMBER(beta) >TO_NUMBER(gamma) OR
                 TO_NUMBER(gamma)>TO_NUMBER(delta) THEN
                 FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||pos||
                 ': Trapecio posibilstico de la etiqueta mal definido: $[a,b,c,d] tal que a<=b<=c<=d.'
                 ||' Actualmente definida como $['||TO_NUMBER(alfa)||','||TO_NUMBER(beta)||','
                 ||TO_NUMBER(gamma)||','||TO_NUMBER(delta)||'].');
              END IF;
            EXCEPTION
              WHEN NO_DATA_FOUND THEN -- Error: Etiqueta no definida en FLD
                FSQL_AUX.inserta_error('ERROR en FIRST en posicin '||pos||
                  ': Etiqueta '||alfa||' definida en FUZZY_OBJECT_LIST para el atributo tipo 2'
                  ||st||columna.atributo||', (OBJ#,COL#)=('||OBJ||','||COL
                  ||'), y no definida en FUZZY_LABEL_DEF.');
                RETURN;
            END;
            -- Sustituir columna por llamada a fcomp_trape (y no a fcomp_label, para > eficiencia)
            IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN
               -- Si son estos comparadores, calculamos el margen y el much
               FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
               IF margen IS NULL THEN RETURN; END IF;
               UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                 atributo='FSQL_FUNCTIONS2.'||fcomp||'_trape('
                 ||alfa||','||TO_CHAR(TO_NUMBER(beta)-TO_NUMBER(alfa))||','
                 ||TO_CHAR(TO_NUMBER(gamma)-TO_NUMBER(delta))||','||delta||','
                 ||OBJ||','||COL||','||st||columna.atributo||'T,'
                 ||st||columna.atributo||'1,'||st||columna.atributo||'2,'
                 ||st||columna.atributo||'3,'||st||columna.atributo||'4,'
                 ||TO_CHAR(margen)||','||TO_CHAR(much)||')'||umbral
                 WHERE indice=columna.indice;
            ELSE
               UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                 atributo='FSQL_FUNCTIONS.'||fcomp||'_trape('
                 ||alfa||','||TO_CHAR(TO_NUMBER(beta)-TO_NUMBER(alfa))||','
                 ||TO_CHAR(TO_NUMBER(gamma)-TO_NUMBER(delta))||','||delta||','
                 ||OBJ||','||COL||','||st||columna.atributo||'T,'
                 ||st||columna.atributo||'1,'||st||columna.atributo||'2,'
                 ||st||columna.atributo||'3,'||st||columna.atributo||'4)'||umbral
                 WHERE indice=columna.indice;
            END IF;
            -- Borramos el '$label'
            UPDATE FSQL_QUERY SET atributo=NULL
              WHERE indice BETWEEN columna.indice+2 AND columna.indice+3;
          EXCEPTION
            WHEN NO_DATA_FOUND THEN -- Error: No existe esa etiqueta
              SELECT posicion INTO pos FROM FSQL_QUERY
                WHERE indice=columna.indice+3;
              FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||pos||
                ': No est definida la etiqueta '||alfa||' para la columna difusa tipo 2 '
                ||st||columna.atributo||', (OBJ#,COL#)=('||OBJ||','||COL
                ||'): Definirla en FUZZY_OBJECT_LIST y FUZZY_LABEL_DEF.');

            WHEN TOO_MANY_ROWS THEN -- Error: Existen varias etiquetas con igual nombre
              SELECT posicion INTO pos FROM FSQL_QUERY
                WHERE indice=columna.indice+3;
              FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||pos||
                ': La etiqueta '||alfa||' est definida mltiples veces en FUZZY_OBJECT_LIST para la columna difusa tipo 2 '
                ||st||columna.atributo||', (OBJ#,COL#)=('||OBJ||','||COL||').');
          END;
        ELSIF f_tipo=1 THEN
          BEGIN
            --  Hallar su identificador (F_ID) en FOL (Si no existe, salta una EXCEPCION)
            SELECT FUZZY_ID INTO F_ID FROM FUZZY_OBJECT_LIST
              WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_NAME=alfa;
            -- Calcular trapecio asociado a esa etiqueta en FLD
            BEGIN
              SELECT ALFA,BETA,GAMMA,DELTA INTO alfa,beta,gamma,delta FROM FUZZY_LABEL_DEF
                WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F_ID;
              -- Comprobar que el trapecio de la etiqueta est bien definido:
              IF TO_NUMBER(alfa) >TO_NUMBER(beta)  OR
                 TO_NUMBER(beta) >TO_NUMBER(gamma) OR
                 TO_NUMBER(gamma)>TO_NUMBER(delta) THEN
                 FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||pos||
                 ': Trapecio posibilstico de la etiqueta mal definido: $[a,b,c,d] tal que a<=b<=c<=d.'
                 ||' Actualmente definida como $['||TO_NUMBER(alfa)||','||TO_NUMBER(beta)||','
                 ||TO_NUMBER(gamma)||','||TO_NUMBER(delta)||'].');
              END IF;
            EXCEPTION
              WHEN NO_DATA_FOUND THEN -- Error: Etiqueta no definida en FLD
                FSQL_AUX.inserta_error('ERROR en FIRST en posicin '||pos||
                  ': Etiqueta '||alfa||' definida en FUZZY_OBJECT_LIST para el atributo tipo 1'
                  ||st||columna.atributo||', (OBJ#,COL#)=('||OBJ||','||COL
                  ||'), y no definida en FUZZY_LABEL_DEF.');
                RETURN;
            END;
            -- Sust. col por funcin fcomp_crisp_trape (y no fcomp_crisp_label, pa + eficiencia)
            IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN
               -- Si son estos comparadores, calculamos el margen y el much
               FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
               IF margen IS NULL THEN RETURN; END IF;
               UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                 atributo='FSQL_FUNCTIONS2.'||fcomp||'_crisp_trape('||st||columna.atributo||','
                 ||alfa||','||TO_CHAR(TO_NUMBER(beta)-TO_NUMBER(alfa))||','
                 ||TO_CHAR(TO_NUMBER(gamma)-TO_NUMBER(delta))||','||delta
                 ||','||TO_CHAR(margen)||','||TO_CHAR(much)||')'||umbral
                 WHERE indice=columna.indice;
            ELSE
               fcompSinN:=FSQL_AUX.Sin_N(fcomp); -- Quitamos la N del comparador si es necesario
               UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                 atributo='FSQL_FUNCTIONS.'||fcompSinN||'_crisp_trape('||st||columna.atributo||','
                 ||alfa||','||TO_CHAR(TO_NUMBER(beta)-TO_NUMBER(alfa))||','
                 ||TO_CHAR(TO_NUMBER(gamma)-TO_NUMBER(delta))||','||delta||')'||umbral
                 WHERE indice=columna.indice;
            END IF;
            -- Borramos el '$label'
            UPDATE FSQL_QUERY SET atributo=NULL
              WHERE indice BETWEEN columna.indice+2 AND columna.indice+3;
          EXCEPTION
            WHEN NO_DATA_FOUND THEN -- Error: No existe esa etiqueta
              SELECT posicion INTO pos FROM FSQL_QUERY
                WHERE indice=columna.indice+3;
              FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||pos||
                ': No est definida la etiqueta '||alfa||' para la columna difusa tipo 1 '
                ||st||columna.atributo||', (OBJ#,COL#)=('||OBJ||','||COL
                ||'): Definirla en FUZZY_OBJECT_LIST y FUZZY_LABEL_DEF.');

            WHEN TOO_MANY_ROWS THEN -- Error: Existen varias etiquetas con igual nombre
              SELECT posicion INTO pos FROM FSQL_QUERY
                WHERE indice=columna.indice+3;
              FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||pos||
                ': La etiqueta '||alfa||' est definida mltiples veces en FUZZY_OBJECT_LIST para la columna difusa tipo 1 '
                ||st||columna.atributo||', (OBJ#,COL#)=('||OBJ||','||COL||').');
          END;
        ELSE -- f_tipo=3 y tenemos fcol FCOMP $label
            IF fcomp<>'FEQ' THEN
               FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
                 ': No est definido el operador '||fcomp||
                 ' sobre atributos tipo 3 como el utilizado: '||st||columna.atributo||'.');
            ELSE -- fcol FEQ $label
              SELECT LEN INTO length1 FROM FUZZY_COL_LIST WHERE OBJ#=OBJ AND COL#=COL;
              -- Hallamos la columna (OBJ,COL) con las etiquetas definidas en FUZZY_NEARNESS_DEF
              BEGIN
                SELECT OBJ#2,COL#2 INTO OBJ2,COL2 FROM FUZZY_COMPATIBLE_COL
                  WHERE OBJ#1=OBJ AND COL#1=COL;
              EXCEPTION
                WHEN NO_DATA_FOUND THEN OBJ2:=OBJ; COL2:=COL;
              END;
              BEGIN
                --  Hallar su identificador (F_ID) en FOL (Si no existe, salta una EXCEPCION)
                SELECT FUZZY_ID INTO F_ID FROM FUZZY_OBJECT_LIST
                  WHERE OBJ#=OBJ2 AND COL#=COL2 AND FUZZY_NAME=alfa;
                IF length1=1 THEN -- Se simplifica la llamada --> Acelera consulta en SQL
                   UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                     atributo='FSQL_FUNCTIONS.FEQ3_label_LEN1('||
                     OBJ2||','||COL2||','||F_ID||','||st||columna.atributo||'T,'||
                     st||columna.atributo||'P1,'||st||columna.atributo||'1)'||umbral
                     WHERE indice=columna.indice;
                ELSE
                   argstipo3:=FSQL_AUX.resto_llamada_tipo3(st||columna.atributo,length1);
                   UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                     atributo='FSQL_FUNCTIONS.FEQ3_label('||
                     OBJ2||','||COL2||','||F_ID||','||length1||','||argstipo3||umbral
                     WHERE indice=columna.indice;
                END IF;
                -- Borramos el '$label'
                UPDATE FSQL_QUERY SET atributo=NULL
                  WHERE indice BETWEEN columna.indice+2 AND columna.indice+3;
              EXCEPTION
                WHEN NO_DATA_FOUND THEN -- Error: No existe esa etiqueta
                  IF OBJ=OBJ2 AND COL=COL2 THEN
                       argstipo3:=NULL;
                  ELSE argstipo3:=', compatible con ('||OBJ2||','||COL2||')';
                  END IF;
                  SELECT posicion INTO pos FROM FSQL_QUERY
                    WHERE indice=columna.indice+3;
                  FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||pos||
                    ': No est definida la etiqueta '||alfa||' para la columna difusa tipo 3 '||
                    st||columna.atributo||', (OBJ#,COL#)=('||OBJ||','||COL||')'||argstipo3
                    ||': Definirla en FUZZY_OBJECT_LIST y en FUZZY_NEARNESS_DEF.');

                WHEN TOO_MANY_ROWS THEN -- Error: Existen varias etiquetas con igual nombre
                  IF OBJ=OBJ2 AND COL=COL2 THEN
                       argstipo3:=NULL;
                  ELSE argstipo3:=', compatible con ('||OBJ2||','||COL2||')';
                  END IF;
                  SELECT posicion INTO pos FROM FSQL_QUERY
                    WHERE indice=columna.indice+3;
                  FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||pos||
                    ': La etiqueta '||alfa||' est definida mltiples veces en FUZZY_OBJECT_LIST para la columna difusa tipo 3 '
                    ||st||columna.atributo||', (OBJ#,COL#)=('||OBJ||','||COL||')'||argstipo3||'.');
              END;
            END IF;
          END IF;
      END IF;

--***** ***** ***** CTES. DIFUSAS: fcol <fcomp> <cte. difusa> ***** ***** *****
    ELSIF token_sig IN ('UNKNOWN','UNDEFINED','NULL') THEN
      IF fcomp<>'FEQ' THEN
         FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
           ': No puede usarse el comparador difuso '||fcomp||
           ' para comparar un atributo ('||st||columna.atributo||
           ') con la constante '||token_sig||': Usar FEQ o IS [NOT].');
      ELSE
        IF (f_tipo=2) OR (f_tipo=3 AND token_sig='UNDEFINED') THEN
           UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
             atributo='FSQL_FUNCTIONS.FEQ_'||token_sig||'('||st||columna.atributo||'T)'
             WHERE indice=columna.indice;
        ELSIF f_tipo=3 THEN -- fcol FEQ [UNKNOWN|NULL]
           SELECT LEN INTO length1 FROM FUZZY_COL_LIST WHERE OBJ#=OBJ AND COL#=COL;
           argstipo3:=FSQL_AUX.resto_llamada_tipo3(st||columna.atributo,length1);
           UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
             atributo='FSQL_FUNCTIONS.FEQ3_UNKNOWN_NULL('||length1||','||argstipo3
             WHERE indice=columna.indice;
        ELSE -- f_tipo=1
          FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
            ': Columna '|| st||columna.atributo||
            ' es tipo 1, y no puede compararse con la constante difusa '||token_sig||'.');
          RETURN;
        END IF;
        -- Borramos la cte
        UPDATE FSQL_QUERY SET atributo=NULL
          WHERE indice=columna.indice+2;
        -- Calculamos la posicin donde debera estar el umbral
        ind:=columna.indice+2;
        LOOP
          ind:=ind+1;
          BEGIN
            SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=ind;
          EXCEPTION
            -- Columna al final de la consulta:
            WHEN NO_DATA_FOUND THEN token_sig:='sin umbral';
          END;
          EXIT WHEN token_sig IN ('sin umbral','THOLD','NUMBRAL','comparador');
        END LOOP;
        -- Calculamos el texto a aadir por el umbral y lo aadimos al final del token anterior
        -- As, son posibles condiciones como: fcol1 FEQ UNDEFINED+4 4.5 --> FEQ...(...)+4 >=4.5
        umbral:=FSQL_AUX.calcula_umbral(ind);
        IF umbral IS NOT NULL THEN
           UPDATE FSQL_QUERY SET atributo=atributo||umbral WHERE indice=ind-1;
        END IF;
      END IF;


--***** ***** ***** EXPRESIN: fcol <fcomp> <expr> ***** ***** *****
    ----> Expresin (puede ser columna o expresin crisp):
    ELSE
      ind:=-1;
      IF token_sig='s.t.' THEN -- Tras el fcomp hay una columna: c FCOMP s.t.c
         ind:=columna.indice+6;
         SELECT indice,posicion,nombre,atributo INTO columna2 FROM FSQL_QUERY
           WHERE indice=ind;
      ELSIF token_sig='t.' THEN
         ind:=columna.indice+4;
         SELECT indice,posicion,nombre,atributo INTO columna2 FROM FSQL_QUERY
           WHERE indice=ind;
      ELSIF token_sig='column' THEN
         ind:=columna.indice+2;
         SELECT indice,posicion,nombre,atributo INTO columna2 FROM FSQL_QUERY
           WHERE indice=ind;
      END IF;

--***** ***** ***** COLUMNA: fcol1 <fcomp> fcol2 ***** ***** *****
      -- Si tras el fcomp hay una columna y esta es difusa...
      IF ind<>-1 AND es_fuzzy_column(columna2,scheme2,tab2,f_tipo2,OBJ2,COL2) THEN
        -- Borramos s.t de la columna 2
        UPDATE FSQL_QUERY SET nombre='ok',atributo=NULL
          WHERE indice BETWEEN columna.indice+2 AND ind-1;
        -- Borramos nombre la columna 2 y la marcamos (en el campo nombre) como tratada:
        UPDATE FSQL_QUERY SET nombre=scheme2||'.'||tab2||'.'||columna2.atributo||'\2\',
          atributo=NULL WHERE indice=ind;
        -- Le aadimos a la columna 2 su esquema.tabla (si procede)
        IF columna2.nombre='t.column' THEN
           columna2.atributo:=tab2||'.'||columna2.atributo;
        ELSIF columna2.nombre='s.t.column' THEN -- caso scheme.tabla.columna
           columna2.atributo:=scheme2||'.'||tab2||'.'||columna2.atributo;
        END IF;

        -- Dependiendo del tipo (1, 2  3) se har un tratamiento u otro:
        IF f_tipo=2 AND (f_tipo2 IN (1,2)) THEN
          IF f_tipo2=2 THEN -- TENEMOS: Fcolumna FCOMP Fcolumna2 (ambas tipo 2)
             -- Sustituimos la columna por la llamada a la funcin correspondiente a fcomp
             IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN -- fcol_t2 MGT fcol_t2
                -- Con estos comparadores hay que calcular dos valores: Margen y Much
                FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
                IF margen IS NULL THEN RETURN; END IF;
                UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                  atributo='FSQL_FUNCTIONS2.'||fcomp||'('||OBJ||','||COL||','||
                  st||columna.atributo||'T,'||
                  st||columna.atributo||'1,'||st||columna.atributo||'2,'||
                  st||columna.atributo||'3,'||st||columna.atributo||'4,'||
                  OBJ2||','||COL2||','||
                  columna2.atributo||'T,'||columna2.atributo||'1,'||columna2.atributo||'2,'||
                  columna2.atributo||'3,'||columna2.atributo||'4,'
                  ||TO_CHAR(margen)||','||TO_CHAR(much)||')'
                  WHERE indice=columna.indice;
             ELSE
                UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                  atributo='FSQL_FUNCTIONS.'||fcomp||'('||OBJ||','||COL||','||
                  st||columna.atributo||'T,'||
                  st||columna.atributo||'1,'||st||columna.atributo||'2,'||
                  st||columna.atributo||'3,'||st||columna.atributo||'4,'||
                  OBJ2||','||COL2||','||
                  columna2.atributo||'T,'||columna2.atributo||'1,'||columna2.atributo||'2,'||
                  columna2.atributo||'3,'||columna2.atributo||'4)'
                  WHERE indice=columna.indice;
             END IF;
             -- Calculamos la posicin donde debera estar el umbral
             LOOP
               ind:=ind+1;
               BEGIN
                 SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=ind;
               EXCEPTION
                 -- Columna al final de la consulta:
                 WHEN NO_DATA_FOUND THEN token_sig:='sin umbral';
               END;
               EXIT WHEN token_sig IN ('sin umbral','THOLD','NUMBRAL','comparador');
             END LOOP;
             -- Calculamos el texto a aadir por el umbral y lo aadimos al final del token anterior
             -- As, son posibles condiciones como: fcol1 FEQ fcol2+5 THOLD 5.5 --> FEQ(...)+5 >=5.5
             umbral:=FSQL_AUX.calcula_umbral(ind);
             IF umbral IS NOT NULL THEN
                UPDATE FSQL_QUERY SET atributo=atributo||umbral WHERE indice=ind-1;
             END IF;
          ELSE -- f_tipo2=1  --> Igual que fcol <fcomp> expr_crisp
            -- NO ES LO MISMO:
            --    1. fcol_t2 <fcomp> fcol_t1+5 THOLD .2 --> FCOMP(fcol_t2,fcol_t1+5)>=.2
            --    2. fcol_t1 <fcomp> fcol_t2+5 THOLD .2 --> FCOMP(fcol_t1,fcol_t2)+5>=.2
            -- Caso 1: Cambiamos la columna por el principio de la llamada a fcomp_crisp
            IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN
                 UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                   atributo='FSQL_FUNCTIONS2.'||fcomp||'_crisp('||columna2.atributo
                   WHERE indice=columna.indice;
            ELSE UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                   atributo='FSQL_FUNCTIONS.'||fcomp||'_crisp('||columna2.atributo
                   WHERE indice=columna.indice;
            END IF;
            -- Calculamos la posicin donde debera estar el umbral
            ind:=columna.indice+2;
            LOOP
              ind:=ind+1;
              BEGIN
                SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=ind;
              EXCEPTION
                -- Columna al final de la consulta:
                WHEN NO_DATA_FOUND THEN token_sig:='sin umbral';
              END;
              EXIT WHEN token_sig IN ('sin umbral','THOLD','NUMBRAL','comparador');
            END LOOP;
            -- Calculamos el texto a aadir por el umbral y lo aadimos
            -- junto con el resto de la llamada a fcomp_crisp en su sitio.
            -- De esta forma, la expresin crisp queda como primer parmetro de fcomp_crisp
            umbral:=FSQL_AUX.calcula_umbral(ind);
            IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN -- fcol_t2 MGT fcol_t1
               -- Con estos comparadores hay que calcular dos valores: Margen y Much
               FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
               IF margen IS NULL THEN RETURN; END IF;
               UPDATE FSQL_QUERY SET atributo=atributo||','||OBJ||','||COL||','
                 ||st||columna.atributo||'T,'
                 ||st||columna.atributo||'1,'||st||columna.atributo||'2,'
                 ||st||columna.atributo||'3,'||st||columna.atributo||'4,'
                 ||TO_CHAR(margen)||','||TO_CHAR(much)||')'||umbral
                 WHERE indice=ind-1;
            ELSE
               UPDATE FSQL_QUERY SET atributo=atributo||','||OBJ||','||COL||','
                 ||st||columna.atributo||'T,'
                 ||st||columna.atributo||'1,'||st||columna.atributo||'2,'
                 ||st||columna.atributo||'3,'||st||columna.atributo||'4)'||umbral
                 WHERE indice=ind-1;
            END IF;
          END IF;

        ELSIF f_tipo=3 AND f_tipo2=3 THEN -- Comparacin de 2 columnas tipo 3
          IF fcomp<>'FEQ' THEN
             FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion
               ||': No est definido el comparador difuso '||fcomp
               ||' sobre atributos tipo 3 ('||st||columna.atributo
               ||'): Usar FEQ o IS [NOT].');
          ELSE -- fcol1 FEQ fcol2 (tipo 3)
            -- Hallamos las longitudes mximas de ambos atributos
            SELECT LEN INTO length1 FROM FUZZY_COL_LIST WHERE OBJ#=OBJ  AND COL#=COL;
            SELECT LEN INTO length2 FROM FUZZY_COL_LIST WHERE OBJ#=OBJ2 AND COL#=COL2;
            -- Para la columna 1: Hallamos la columna (OBJ,COL) con las etiquetas definidas en FUZZY_NEARNESS_DEF
            BEGIN SELECT OBJ#2,COL#2 INTO OBJ,COL FROM FUZZY_COMPATIBLE_COL
                    WHERE OBJ#1=OBJ AND COL#1=COL;
            EXCEPTION WHEN NO_DATA_FOUND THEN NULL;
            END;
            -- Para la columna 2: Hallamos la columna (OBJ,COL) con las etiquetas definidas en FUZZY_NEARNESS_DEF
            BEGIN SELECT OBJ#2,COL#2 INTO OBJ2,COL2 FROM FUZZY_COMPATIBLE_COL
                    WHERE OBJ#1=OBJ2 AND COL#1=COL2;
            EXCEPTION WHEN NO_DATA_FOUND THEN NULL;
            END;
            IF OBJ<>OBJ2 OR COL<>COL2 THEN
               FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
                 ': No se pueden comparar atributos tipo 3 incompatibles ('||
                 st||columna.atributo||' y '||columna2.atributo||
                 '): Se debe indicar su compatibilidad en la tabla FUZZY_COMPATIBLE_COL.');
            ELSE -- Tenemos 2 columnas tipo 3 compatibles
              -- Hallamos el resto de argumentos de la columna 1 y le quitamos el ltimo carcter: ')'
              argstipo3:=FSQL_AUX.resto_llamada_tipo3(st||columna.atributo,length1);
              argstipo3:=SUBSTR(argstipo3,1,LENGTH(argstipo3)-1);
              -- Hallamos el resto de argumentos de la columna 2
              argstipo3:=argstipo3||','||FSQL_AUX.resto_llamada_tipo3(columna2.atributo,length2);
              -- Sustituimos la columna por la llamada a la funcin correspondiente a fcomp
              UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                atributo='FSQL_FUNCTIONS.FEQ3('||OBJ||','||COL||','||
                length1||','||length2||','||argstipo3
                WHERE indice=columna.indice;
              -- Calculamos la posicin donde debera estar el umbral
              LOOP
                ind:=ind+1;
                BEGIN
                  SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=ind;
                EXCEPTION
                  WHEN NO_DATA_FOUND THEN token_sig:='sin umbral'; -- Columna al final de la consulta
                END;
                EXIT WHEN token_sig IN ('sin umbral','THOLD','NUMBRAL','comparador');
              END LOOP;
              -- Calculamos el texto a aadir por el umbral y lo aadimos al final del token anterior
              -- As, son posibles condiciones como: fcol1 FEQ fcol2+5 THOLD 5.5 --> FEQ(...)+5 >=5.5
              umbral:=FSQL_AUX.calcula_umbral(ind);
--              dbms_output.put_line(ind||','||umbral);
              IF umbral IS NOT NULL THEN
                 UPDATE FSQL_QUERY SET atributo=atributo||umbral WHERE indice=ind-1;
              END IF;
            END IF;
          END IF;
        ELSIF f_tipo=1 THEN
          IF f_tipo2=1 THEN
             IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN -- fcol_t1 MGT fcol2_t1
                -- Comparacin de dos atributos tipo 1, o un tipo1
                -- y una expr_crisp, con los comparadores (de Posibilidad o Necesidad)
                -- "Mucho Mayor Que" (MGT,NMGT) o "Mucho Menor Que" (MLT,NMLT):
                UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                  atributo='FSQL_FUNCTIONS2.'||fcomp||'_t1_t1('
                  ||st||columna.atributo||','||columna2.atributo
                  WHERE indice=columna.indice;
                -- Calculamos la posicin donde debera estar el umbral
                ind:=columna.indice+2;
                LOOP ind:=ind+1;
                  BEGIN
                    SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=ind;
                  EXCEPTION
                    WHEN NO_DATA_FOUND THEN token_sig:='sin umbral'; -- Columna al final de la consulta
                  END;
                  EXIT WHEN token_sig IN ('sin umbral','THOLD','NUMBRAL','comparador');
                END LOOP;
                umbral:=FSQL_AUX.calcula_umbral(ind);
                FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
                IF margen IS NULL THEN RETURN; END IF;
                UPDATE FSQL_QUERY SET atributo=atributo||','
                  ||TO_CHAR(margen)||','||TO_CHAR(much)||')'||umbral
                  WHERE indice=ind-1;
             ELSE
               -- Comparacin Fcol1 FCOMP Fcol1 (expr_crisp): En este caso se difumina
               -- la parte derecha de la comparacin, como si fuera un APROX
               -- y se usa la funcin FCOMP_t1_t1
               fcompSinN:=FSQL_AUX.Sin_N(fcomp); -- Quitamos la N del comparador si es necesario
               UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                 atributo='FSQL_FUNCTIONS.'||fcompSinN||'_t1_t1('||st||columna.atributo
                          ||','||columna2.atributo
                 WHERE indice=columna.indice;
               -- Calculamos la posicin donde debera estar el umbral
               ind:=columna.indice+2;
               LOOP
                 ind:=ind+1;
                 BEGIN
                   SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=ind;
                 EXCEPTION
                   WHEN NO_DATA_FOUND THEN token_sig:='sin umbral'; -- Columna al final de la consulta
                 END;
                 EXIT WHEN token_sig IN ('sin umbral','THOLD','NUMBRAL','comparador');
               END LOOP;
               -- Calculamos el texto a aadir por el umbral y lo aadimos al final del
               -- token anterior, junto con el resto de la llamada a la funcin
               -- De esta forma, la expresin crisp queda como segundo parmetro de fcomp_t1_t1
               umbral:=FSQL_AUX.calcula_umbral(ind);
               FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
               IF margen IS NULL THEN RETURN; END IF;
               UPDATE FSQL_QUERY SET atributo=atributo||','||TO_CHAR(margen)||')'||umbral
                 WHERE indice=ind-1;
-- Cuando comparamos 2 valores crisp, podemos dar un error, aunque es mejor
-- difuminar el segundo crisp a un aprox y compararlos normalmente.
--               IF    fcomp='FEQ'  THEN alfa:='= ';
--               ELSIF fcomp='FGT'  THEN alfa:='> ';  ELSIF fcomp='FLT'  THEN alfa:='< ';
--               ELSIF fcomp='FGEQ' THEN alfa:='>= '; ELSIF fcomp='FLEQ' THEN alfa:='<= ';
--               ELSE  alfa:='';
--               END IF;
--               FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
--                 ': Atributos '||st||columna.atributo||' y '||columna2.atributo||
--                 ' son tipo 1. Usar comparador crisp '||alfa||'para esta comparacin.');
             END IF;

          ELSIF f_tipo2=3 THEN
             FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
               ': No se pueden comparar atributos tipo 1 ('||st||columna.atributo||
               ') y tipo 3 ('||columna2.atributo||').');
          ELSE -- f_tipo2=2
            -- NO ES LO MISMO:
            --    1. fcol_t2 <fcomp> fcol_t1+5 THOLD .2 --> FCOMP(fcol_t2,fcol_t1+5)>=.2
            --    2. fcol_t1 <fcomp> fcol_t2+5 THOLD .2 --> FCOMP(fcol_t1,fcol_t2)+5>=.2
            -- Estamos en el caso 2: Cambiamos la columna por la llamada a 'fcomp_crisp'
            -- Adems, las funciones 'fcomp_crisp' no son conmutativas (salvo FEQ).
            -- NO ES LO MISMO:
            --    1. 'fcol_t2 FGT fcol_t1' --> Usa funcin 'FGT_crisp'
            --    2. 'fcol_t1 FGT fcol_t2' --> Usa funcin 'FGT_t1_t2'
            -- Estamos tambin en el caso 2:
            IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN
               -- Si son estos comparadores, calculamos el margen y el much
               FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
               IF margen IS NULL THEN RETURN; END IF;
               -- Usamos la variable st, para no declarar otra, y le
               -- asignamos la 'coletilla', para las llamadas a MGT/MLT/NMGT/NMLT
               UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                 atributo='FSQL_FUNCTIONS2.'||fcomp||'_t1_t2('
                 ||st||columna.atributo||','||OBJ2||','||COL2||','
                 ||columna2.atributo||'T,'
                 ||columna2.atributo||'1,'||columna2.atributo||'2,'
                 ||columna2.atributo||'3,'||columna2.atributo||'4'
                 ||','||TO_CHAR(margen)||','||TO_CHAR(much)||')'
                 WHERE indice=columna.indice;
            ELSE -- Si no es MGT/MLT/NMGT/NMLT, no lleva esa 'coletilla'
               fcompSinN:=FSQL_AUX.Sin_N(fcomp); -- Quitamos la N del comparador si es necesario
               UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                 atributo='FSQL_FUNCTIONS.'||fcompSinN||'_t1_t2('
                 ||st||columna.atributo||','||OBJ2||','||COL2||','
                 ||columna2.atributo||'T,'
                 ||columna2.atributo||'1,'||columna2.atributo||'2,'
                 ||columna2.atributo||'3,'||columna2.atributo||'4)'
                 WHERE indice=columna.indice;
            END IF;
            -- Calculamos la posicin donde debera estar el umbral
            LOOP
              ind:=ind+1;
              BEGIN
                SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=ind;
              EXCEPTION
                WHEN NO_DATA_FOUND THEN token_sig:='sin umbral'; -- Columna al final de la consulta
              END;
              EXIT WHEN token_sig IN ('sin umbral','THOLD','NUMBRAL','comparador');
            END LOOP;
            -- Calculamos el texto a aadir por el umbral y lo aadimos al final del token anterior
            umbral:=FSQL_AUX.calcula_umbral(ind);
            IF umbral IS NOT NULL THEN
               UPDATE FSQL_QUERY SET atributo=atributo||umbral WHERE indice=ind-1;
            END IF;
          END IF;
        ELSE
          FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
            ': Comparacin difusa '||fcomp||
            ' con columnas de distintos tipos ('||st||columna.atributo||
            ' es tipo '||f_tipo||' y '||columna2.atributo||
            ' es tipo '||f_tipo2||').');
          -- Marcamos la columna2 para no tratarla de nuevo posteriormente:
          UPDATE FSQL_QUERY SET nombre='ok' WHERE indice=columna2.indice;
        END IF;

--***** ***** ***** EXPRESIN CRISP: fcol <fcomp> <expr crisp> ***** ***** *****
      ELSE -- Entonces tenemos: col FCOMP expr_crisp (si no es crisp: Error)
        IF f_tipo=3 THEN -- Error: No se puede comparar una columna tipo 3 con una expresin
           FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
             ': No se puede comparar un atributo tipo 3 ('||st||columna.atributo||
             ') con una expresin crisp.');
           IF fcomp<>'FEQ' THEN
              FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion
                ||': No est definido el comparador difuso '||fcomp
                ||' sobre atributos tipo 3 ('||st||columna.atributo
                ||'): Usar FEQ o IS [NOT].');
           END IF;
        ELSIF f_tipo=2 THEN
          -- Cambiamos la columna por el principio de la llamada a fcomp_crisp
          IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN
               UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                 atributo='FSQL_FUNCTIONS2.'||fcomp||'_crisp('
                 WHERE indice=columna.indice;
          ELSE UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
                 atributo='FSQL_FUNCTIONS.'||fcomp||'_crisp('
                 WHERE indice=columna.indice;
          END IF;
          -- Calculamos la posicin donde debera estar el umbral
          ind:=columna.indice+2;
          LOOP
            ind:=ind+1;
            BEGIN
              SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=ind;
            EXCEPTION
              WHEN NO_DATA_FOUND THEN token_sig:='sin umbral'; -- Columna al final de la consulta
            END;
            EXIT WHEN token_sig IN ('sin umbral','THOLD','NUMBRAL','comparador');
          END LOOP;
          -- Calculamos el texto a aadir por el umbral y lo aadimos al final del
          -- token anterior, junto con el resto de la llamada a fcomp_crisp
          -- De esta forma, la expresin crisp queda como primer parmetro de fcomp_crisp
          umbral:=FSQL_AUX.calcula_umbral(ind);
          IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN -- fcol_t2 MGT expr_crisp
             -- Con estos comparadores hay que calcular dos valores: Margen y Much
             FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
             IF margen IS NULL THEN RETURN; END IF;
             UPDATE FSQL_QUERY SET atributo=atributo||','||OBJ||','||COL||','||
               st||columna.atributo||'T,'||
               st||columna.atributo||'1,'||st||columna.atributo||'2,'||
               st||columna.atributo||'3,'||st||columna.atributo||'4,'
               ||TO_CHAR(margen)||','||TO_CHAR(much)||')'||umbral
               WHERE indice=ind-1;
          ELSE
             UPDATE FSQL_QUERY SET atributo=atributo||','||OBJ||','||COL||','||
               st||columna.atributo||'T,'||
               st||columna.atributo||'1,'||st||columna.atributo||'2,'||
               st||columna.atributo||'3,'||st||columna.atributo||'4)'||umbral
               WHERE indice=ind-1;
          END IF;
        ELSE -- f_tipo=1
          IF fcomp IN ('MGT','MLT','NMGT','NMLT') THEN -- fcol_t1 MGT expr_crisp
             -- Comparacin de dos atributos tipo 1 con los comparadores (de Posibilidad o Necesidad)
             -- "Mucho Mayor Que" (MGT/NMGT) o "Mucho Menor Que" (MLT/NMLT):
             -- Cambiamos la columna por el principio de la llamada a fcomp_t1_t1
             UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
               atributo='FSQL_FUNCTIONS2.'||fcomp||'_t1_t1('||st||columna.atributo||','
               WHERE indice=columna.indice;
             -- Calculamos la posicin donde debera estar el umbral
             ind:=columna.indice+2;
             LOOP
               ind:=ind+1;
               BEGIN
                 SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=ind;
               EXCEPTION
                 WHEN NO_DATA_FOUND THEN token_sig:='sin umbral'; -- Columna al final de la consulta
               END;
               EXIT WHEN token_sig IN ('sin umbral','THOLD','NUMBRAL','comparador');
             END LOOP;
             -- Calculamos el texto a aadir por el umbral y lo aadimos al final del
             -- token anterior, junto con el resto de la llamada a fcomp_t1_t1
             -- De esta forma, la expresin crisp queda como segundo parmetro de fcomp_t1_t1
             umbral:=FSQL_AUX.calcula_umbral(ind);
             FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
             IF margen IS NULL THEN RETURN; END IF;
             UPDATE FSQL_QUERY SET atributo=atributo
               ||','||TO_CHAR(margen)||','||TO_CHAR(much)||')'||umbral
               WHERE indice=ind-1;
          ELSE
             -- Comparacin Fcol1 FCOMP expr_crisp: En este caso se difumina
             -- la parte derecha de la comparacin, como si fuera un APROX
             fcompSinN:=FSQL_AUX.Sin_N(fcomp); -- Quitamos la N del comparador si es necesario
             UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
               atributo='FSQL_FUNCTIONS.'||fcompSinN||'_t1_t1('||st||columna.atributo||','
               WHERE indice=columna.indice;
             -- Calculamos la posicin donde debera estar el umbral
             ind:=columna.indice+2;
             LOOP
               ind:=ind+1;
               BEGIN
                 SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=ind;
               EXCEPTION
                 WHEN NO_DATA_FOUND THEN token_sig:='sin umbral'; -- Columna al final de la consulta
               END;
               EXIT WHEN token_sig IN ('sin umbral','THOLD','NUMBRAL','comparador');
             END LOOP;
             -- Calculamos el texto a aadir por el umbral y lo aadimos al final del
             -- token anterior, junto con el resto de la llamada a la funcin
             -- De esta forma, la expresin crisp queda como segundo parmetro de fcomp_t1_t1
             umbral:=FSQL_AUX.calcula_umbral(ind);
             FSQL_AUX.Calcula_Margen_Much(OBJ,COL,pos,st,columna.atributo,margen,much);
             IF margen IS NULL THEN RETURN; END IF;
             UPDATE FSQL_QUERY SET atributo=atributo
               ||','||TO_CHAR(margen)||')'||umbral
               WHERE indice=ind-1;
-- Cuando comparamos 2 valores crisp, podemos dar un error, aunque es mejor
-- difuminar el segundo crisp a un aprox y compararlos normalmente.
--             IF    fcomp='FEQ' THEN alfa:='= ';
--             ELSIF fcomp='FGT' THEN alfa:='> ';   ELSIF fcomp='FLT' THEN alfa:='< ';
--             ELSIF fcomp='FGEQ' THEN alfa:='>= '; ELSIF fcomp='FLEQ' THEN alfa:='<= ';
--             ELSE  alfa:='';
--             END IF;
--             FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
--               ': Atributo '||st||columna.atributo||
--               ' es tipo 1. Usar comparador crisp '||alfa||'para esta comparacin.');
          END IF;
        END IF;
      END IF;

    END IF;

--***** ***** ***** Despus de un fcomparador: <expr> <fcomp> fcol ***** ***** *****
  ELSIF token_ant IN ('FGT','FLT','FEQ','FGEQ','FLEQ','MGT','MLT',
                      'NFGT','NFLT','NFEQ','NFGEQ','NFLEQ','NMGT','NMLT') THEN
    FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||pos||
      ': Comparador difuso '||token_ant||
      ' debe tener una columna difusa justo a su izquierda (tipo 1, 2  3). Formato: <fuzzy_column> '||
      token_ant||' <expr>.');
    -- Esto se debe modificar para que no sea error y conseguir consultas ms flexibles...
    -- Habra que examinar operando antes del fcomparador:
    --  * No puede ser una fcolumn (se hubiera tratado antes)
    --      * Delimitado por: NOT, AND, OR, WHERE o un ( o { sin su pareja antes del fcomparador

--***** ***** ***** ***** ***** fcol IS <cte. difusa> ***** ***** ***** ***** *****
  ELSIF token_sig='IS' THEN  -- fcol IS [UNKNOWN|UNDEFINED|NULL]
    -- Con el 'IS' se busca los atributos que valen exactamente esa cte.
    -- y no su compatibilidad con la cte. (eso sera con FEQ).
    -- Su tratamiento es independiente del tipo difuso (2  3) del atributo.
    SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=columna.indice+2;
    IF token_sig='NOT' THEN
       SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=columna.indice+3;
       IF f_tipo=1 THEN
          IF token_sig='NULL' THEN -- Se refiere al NULL de Oracle (no al difuso)
             RETURN;
          ELSE
             FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
               ': Atributo tipo 1 '||st||columna.atributo||
               ' no puede compararse mediante IS NOT con la constante difusa '||token_sig||'.');
             RETURN;
          END IF;
       END IF;
       -- Sustituir columna por la comparacin pertinente
       IF token_sig='UNKNOWN' THEN
         UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
           atributo=st||columna.atributo||'T<>0'
           WHERE indice=columna.indice;
       ELSIF token_sig='UNDEFINED' THEN
         UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
           atributo=st||columna.atributo||'T<>1'
           WHERE indice=columna.indice;
       ELSE --token_sig='NULL'
         UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
           atributo=st||columna.atributo||'T<>2'
           WHERE indice=columna.indice;
       END IF;
       -- Borramos el UNKNOWN, UNDEFINED o NULL
       UPDATE FSQL_QUERY SET atributo=NULL WHERE indice=columna.indice+3;
     ELSIF f_tipo=1 THEN
       IF token_sig='NULL' THEN -- Se refiere al NULL de Oracle (no al difuso)
          RETURN;
       ELSE
          FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
            ': Atributo tipo 1 '||st||columna.atributo||
            ' no puede compararse mediante IS con la constante difusa '||token_sig||'.');
          RETURN;
       END IF;
    ELSIF token_sig='UNKNOWN' THEN -- Sustituir columna por la comparacin pertinente
      UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
        atributo=st||columna.atributo||'T=0'
        WHERE indice=columna.indice;
    ELSIF token_sig='UNDEFINED' THEN
      UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
        atributo=st||columna.atributo||'T=1'
        WHERE indice=columna.indice;
    ELSE --token_sig='NULL'
      UPDATE FSQL_QUERY SET nombre=scheme||'.'||tab||'.'||columna.atributo||'\',
        atributo=st||columna.atributo||'T=2'
        WHERE indice=columna.indice;
    END IF;
    -- Borramos 2 tokens: El 'IS ...' o el 'IS NOT'
    UPDATE FSQL_QUERY SET atributo=NULL
      WHERE indice BETWEEN columna.indice+1 AND columna.indice+2;

--***** ***** ***** ***** ***** En otro caso: Error ***** ***** ***** ***** *****
  ELSE -- Si la columna difusa no se ha tratado en los casos anteriores y no es tipo 1: Tres opciones:
       --       1. Dar un ERROR, indicando que la columna est mal situada porque las columnas difusas
       --          no admiten operaciones aritmticas, ni ser arg. de funciones (excepto CDEG).
       --       2. Usar el TIPO del valor DIFUSO de cada tupla (aadiendo una 'T' a dicho atributo).
       --       3. Tratarla como si estuviera en la select_list (llamada a funcin SHOW...)
       -- La opcin a efectuar se puede modificar en la vista FSQL_OPTIONS, con OPCION='TRATA_FUZZY_ATRIB'
       -- Las opciones vlidas para el campo VALOR son:
       -- ERROR (opcin 1.), TIPO_VALOR (opcin 2.) y REPRESENT (opcin 3. y por defecto)
    IF f_tipo<>1 THEN
       IF TRATA_FUZZY_ATRIB='ERROR' THEN -- Opcin 1: Dar un error
          FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
            ': El atributo difuso '||st||columna.atributo||' est mal situado.');
       ELSIF TRATA_FUZZY_ATRIB='TIPO_VALOR' THEN -- Opcin 2: Tipo del valor
         -- ESTABLECER QUE si un atributo difuso est colocado en un lugar extrao nos referimos al TIPO del valor
         -- DIFUSO de cada tupla (de 0 a 7 para atributos difusos tipo 2 y de 0 a 4 para atributos difusos tipo 3).
         -- Esto es til, por ejemplo en la clusula 'ORDER BY col_t2' que pasa a 'ORDER BY col_t2T':
         UPDATE FSQL_QUERY SET atributo=atributo||'T' WHERE indice=columna.indice;
       ELSE -- Opcin 3 y por defecto (TRATA_FUZZY_ATRIB='REPRESENT')
         -- El problema del caso 2 es que slo ordenara por tipo y no por contenido, con lo cual no quedara
         -- ordenado como uno espera. Solucin: Ordenar sobre la representacin grfica de esa columna:
         -- Inconveniente: No se permiten cosas como 'SQRT(col_t2)' --> Error de Oracle: ORA-01722 (invalid number)
         -- lo cual es, aunque restrictivo, razonable. Si estn permitidas, sin embargo, cosas como 'SQRT(col_t2T)'
         FSQL_AUX.fuzzy_column_en_sl(st||columna.atributo,columna.indice,f_tipo,OBJ,COL,'');
       END IF;
    END IF;
  END IF;
END trata_fuzzy_column;

----------------------------------------------------------------------------------------------
-- Busca el parntesis derecho ')' que cierra la llamada a una funcin
-- Empieza buscando en el atributo de la tupla condicion y sigue en las
-- tuplas sucesivas.
-- Devuelve: tupla  --> Nmero de tupla en la que lo encontr (0 si es en condicion)
--           offset --> Posicin del parntesis en dicha tupla
----------------------------------------------------------------------------------------------
PROCEDURE busca_rparent(condicion IN QUERYTYPE,
                        tupla IN OUT INTEGER,
                        offset   OUT INTEGER)
IS
  cadena FSQL_QUERY.atributo%TYPE:=condicion.atributo;
  pos  INTEGER;
  cont INTEGER:=0;
  long INTEGER;
  ch   CHAR:='';
  encontrado BOOLEAN:=FALSE;
BEGIN
  tupla:=0;
  WHILE NOT encontrado LOOP
    pos:=1;
    long:=LENGTH(cadena);
--  dbms_output.put_line(cadena||'-->'||long);
    LOOP
      ch:=SUBSTR(cadena,pos,1);
      IF    ch='(' THEN cont:=cont+1; -- Nos saltamos los (...) intermedios que veamos
      ELSIF ch=')' THEN cont:=cont-1;
      END IF;
--  dbms_output.put_line(ch||'-'||tupla||'-'||pos||'-'||cont);
      -- Salimos cuando llegamos al final o cuando encontramos el ')' buscado
      EXIT WHEN pos>=long OR (ch=')' AND cont=0);
      pos:=pos+1;
    END LOOP;

--  dbms_output.put_line('pos='||pos||' cont='||cont);
    IF ch=')' AND cont=0 THEN -- Hemos encontrado el ')'
       offset:=pos;
       encontrado:=TRUE;
    ELSE -- Hemos llegado al final sin encontrar el ')': Buscamos en la siguiente tupla<>NULL
       LOOP
         tupla:=tupla+1;
--  dbms_output.put_line(condicion.indice||'+'||tupla);
         SELECT atributo INTO cadena FROM FSQL_QUERY WHERE indice=condicion.indice+tupla;
         EXIT WHEN cadena IS NOT NULL;
--  dbms_output.put_line('es null!!!!');
       END LOOP;
    END IF;
  END LOOP;
END busca_rparent;

----------------------------------------------------------------------------------------------
-- Devuelve el Grado de cumplimiento en una condicin difusa para una columna
-- Entrada: Tupla de FSQL_QUERY donde estaba la columna en la clusula WHERE.
-- En general, es la condicin que est puesta, eliminando el comparador crisp
-- y el umbral de cumplimiento.
-- Requisito: condicion.atributo IS NOT NULL (que no sea col2 en: col1 <fcomp> col2)
----------------------------------------------------------------------------------------------
FUNCTION Grado_Condicion(condicion IN QUERYTYPE,
                         tupla IN OUT INTEGER) RETURN VARCHAR2 IS
  -- Grado de cumplimiento en una condicin difusa (condicin sin umbral):
  grado_cond FSQL_QUERY.atributo%TYPE:='';
  atrib      FSQL_QUERY.atributo%TYPE;
  offset     INTEGER;
BEGIN
  -- Si empieza por una llamada al paquete de funciones...
  IF condicion.atributo LIKE 'FSQL_FUNCTIONS%' THEN
     busca_rparent(condicion,tupla,offset);
--   dbms_output.put_line(tupla||':'||offset);
     IF tupla=0 THEN -- El ) que cierra la llamada est en esta misma tupla
        RETURN SUBSTR(condicion.atributo,1,offset);
     ELSE
        grado_cond:=condicion.atributo; -- Asignamos la 1 parte
        FOR i IN 1..tupla-1 LOOP -- Bucle para concatenar la parte de el medio
          SELECT atributo INTO atrib FROM FSQL_QUERY WHERE indice=condicion.indice+i;
          grado_cond:=grado_cond||atrib;
        END LOOP;
        SELECT atributo INTO atrib FROM FSQL_QUERY WHERE indice=condicion.indice+tupla;
        RETURN grado_cond||SUBSTR(atrib,1,offset); -- Concatenamos la parte final
     END IF;
  ELSE -- Estamos en cualquier otro caso crisp
    tupla:=0;
    RETURN '1';
  END IF;
END Grado_Condicion;

----------------------------------------------------------------------------------------------
-- Devuelve el Grado de cumplimiento en una condicin crisp con una columna
-- Entrada: Tupla de FSQL_QUERY donde estaba la columna en la clusula WHERE.
-- Se traduce por una llamada a FSQL_FUNCTIONS.CompCrisp(op1, comparador, op2)
-- Si no encuentra el comparador crisp devuelve NULL
----------------------------------------------------------------------------------------------
FUNCTION Grado_CondCrisp(condicion IN     QUERYTYPE,
                         tupla     IN OUT INTEGER) RETURN VARCHAR2 IS
  token QUERYTYPE;
  op1   FSQL_QUERY.atributo%TYPE:=''; -- op1 comparador op2 (en una comparacion crisp)
  op2   FSQL_QUERY.atributo%TYPE:='';
  comparador VARCHAR2(2);
BEGIN
  -- Buscamos hacia atras el principio de la condicin crisp
  tupla:=condicion.indice-1;
  LOOP
    BEGIN SELECT indice,posicion,nombre,atributo INTO token FROM FSQL_QUERY
            WHERE indice=tupla;
    EXCEPTION WHEN NO_DATA_FOUND THEN token.nombre:='fin';
    END;
    EXIT WHEN token.nombre IN ('WHERE','NOT','AND','OR','(','{');
    tupla:=tupla-1;
  END LOOP;

  LOOP -- Copiamos el operador 1 en op1
    tupla:=tupla+1;
    BEGIN SELECT indice,posicion,nombre,atributo INTO token FROM FSQL_QUERY
            WHERE indice=tupla;
    EXCEPTION WHEN NO_DATA_FOUND THEN
      -- Si no hemos encontrado el comparador crisp, es que NO es una condicin crisp, sino
      -- que era una expresin aritmtica. Ej: WHERE Distancia + Distancia FEQ #3 THOLD 40
      RETURN NULL;
    END;
    EXIT WHEN token.nombre='comparador';
    op1:=op1||token.atributo;
  END LOOP;
  comparador:=token.atributo;

  LOOP -- Copiamos el operador 2 en op2
    tupla:=tupla+1;
    BEGIN SELECT indice,posicion,nombre,atributo INTO token FROM FSQL_QUERY
            WHERE indice=tupla;
    EXCEPTION WHEN NO_DATA_FOUND THEN token.nombre:='fin';
    END;
    EXIT WHEN token.nombre IN ('AND','OR','fin','}',')',';',
                               'GROUP','CONNECT','START','UNION','INTERSECT','MINUS','ORDER');
    op2:=op2||token.atributo;
  END LOOP;
--  dbms_output.put_line(op1||comparador||op2);
  tupla:=tupla-condicion.indice-1;
  RETURN 'FSQL_FUNCTIONS.CondCrisp('||op1||','||CHR(39)||comparador||CHR(39)||','||op2||')';
END Grado_CondCrisp;

----------------------------------------------------------------------------------------------
-- Lee las opciones de configuracin del usuario para el Servidor FSQL:
-- 1. Funciones a utilizar en los operadores lgicos NOT, AND y OR.
--    Estas estn almacenadas, para cada usuario, en la vista FSQL_OPTIONS.
--    En caso de que no estn almacenadas ah, se usan las funciones por defecto.
--            Operacin Logica  Funcin por defecto Otro caso
--            ----------------  ------------------- ---------
--            NOT a         1-a             F_NOT(a)
--            a AND b           LEAST(a,b)          F_AND(a,b)
--            a OR  b           GREATEST(a,b)       F_OR (a,b)
--    Requisitos: Los parmetros almacenados en el campo FSLQ_OPTIONS.Valor
--                debe ser funciones alcanzables por el Usuario y deben tener 1, 2 y 2
--                argumentos respectivamente para NOT, AND y OR.
-- 2. Tipo de tratamiento para los atributos difusos en situacin inusual:
--    Valores vlidos:
--            ERROR      --> Genera un error
--            TIPO_VALOR --> Utiliza el tipo del valor del atributo: Tipo NUMBER(1)
--                           (concatenar 'T' al nombre del atributo)
--            REPRESENT  --> Utiliza la representacin grfica del atributo: Tipo VARCHAR2
--                           (opcin por defecto)
----------------------------------------------------------------------------------------------
PROCEDURE Lee_OPTIONS IS
BEGIN
  -- Funcin a utilizar con NOT:
  BEGIN SELECT Valor INTO F_NOT FROM FSQL_OPTIONS WHERE OPCION='NOT';
  EXCEPTION WHEN NO_DATA_FOUND THEN F_NOT:='1-'; -- El '1-' indica que usaremos la opcin por defecto
  END;
  -- Funcin a utilizar con AND:
  BEGIN SELECT Valor INTO F_AND FROM FSQL_OPTIONS WHERE OPCION='AND';
  EXCEPTION WHEN NO_DATA_FOUND THEN F_AND:='LEAST';
  END;
  -- Funcin a utilizar con OR:
  BEGIN SELECT Valor INTO F_OR FROM FSQL_OPTIONS WHERE OPCION='OR';
  EXCEPTION WHEN NO_DATA_FOUND THEN F_OR:='GREATEST';
  END;
  -- Tratamiento a atributos difusos en posiciones inusuales
  BEGIN SELECT Valor INTO TRATA_FUZZY_ATRIB FROM FSQL_OPTIONS WHERE OPCION='TRATA_FUZZY_ATRIB';
  EXCEPTION WHEN NO_DATA_FOUND THEN TRATA_FUZZY_ATRIB:='REPRESENT';
  END;
END Lee_OPTIONS;

----------------------------------------------------------------------------------------------
-- Cambia operadores lgicos NOT, AND y OR de la tabla CDEG, entre las posiciones inicio y fin,
-- cuando se trata de un CDEG(*) y hay que tratar todas las condiciones.
-- Utiliza las funciones calculadas por Lee_OPTIONS;
-- Requisito: Entre las posiciones inicio+1 y fin-1 no deben existir parntesis
----------------------------------------------------------------------------------------------
PROCEDURE cambia_oper_logicos(CDEG     IN OUT TABLA_CDEG,
                              max_CDEG IN OUT INTEGER,
                              inicio   IN     FSQL_QUERY.indice%TYPE,
                              fin      IN OUT FSQL_QUERY.indice%TYPE) IS
  offset INTEGER:=0; -- Desplazamiento de los valores de la tabla (a partir de fin)
  pos    FSQL_QUERY.posicion%TYPE:=inicio;
BEGIN
  -- Antes de cambiar operadores lgicos vemos si hay parntesis en la primera y ltima posiciones
  IF CDEG(inicio) IN ('(','{') THEN
     fin:=fin-2;
     FOR i IN inicio..fin LOOP
       CDEG(i):=CDEG(i+1);
     END LOOP;
     offset:=2; -- Todo se desplazar 2 posiciones (las de los 2 parntesis)
  END IF;

  -- Cambiamos OPERADORES LGICOS en 3 pasos, por orden de precedencia de los operadores lgicos:
  -- 1. Buscamos negaciones: NOT
  WHILE pos<=fin LOOP
    IF CDEG(pos)='NOT' THEN
       IF F_NOT='1-' THEN -- Valor por defecto para la funcin del NOT
          IF    CDEG(pos+1)='1' THEN
                CDEG(pos):='0';
          ELSIF CDEG(pos+1)='0' THEN -- Caso: NOT (NOT algo_que_es_1)
                CDEG(pos):='1';
          ELSE  CDEG(pos):='(1-('||CDEG(pos+1)||'))';
          END IF;
       ELSE -- Se est usando otra funcin para evaluar el NOT
          CDEG(pos):=F_NOT||'('||CDEG(pos+1)||'))';
       END IF;
       fin:=fin-1;
       offset:=offset+1;
       FOR i IN (pos+1)..fin LOOP -- Desplazo el resto hacia abajo una posicin:
         CDEG(i):=CDEG(i+1);
       END LOOP;
    END IF;
    pos:=pos+1;
  END LOOP;

  -- 2. Buscamos conjunciones: AND
  pos:=inicio+1; -- En la primera posicin no puede haber un AND (es op. binario)
  WHILE pos<=fin LOOP
    IF CDEG(pos)='AND' THEN
       pos:=pos-1;
       IF F_AND='LEAST' THEN
          -- El mnimo de 1 y 'algo' es: ese 'algo'
          -- El mnimo de 0 y 'algo' es: 0
          IF    CDEG(pos+2)='1' OR CDEG(pos)='0' THEN
                NULL;
          ELSIF CDEG(pos+2)='0' THEN
                CDEG(pos):='0';
          ELSIF CDEG(pos)='1' THEN
                CDEG(pos):=CDEG(pos+2);
          ELSE  CDEG(pos):='LEAST('||CDEG(pos)||','||CDEG(pos+2)||')';
          END IF;
       ELSE -- Se est usando otra funcin para evaluar el AND
          CDEG(pos):=F_AND||'('||CDEG(pos)||','||CDEG(pos+2)||')';
       END IF;
       FOR i IN pos+1..(fin-2) LOOP -- Desplazo el resto hacia abajo dos posiciones:
         CDEG(i):=CDEG(i+2);
       END LOOP;
       fin:=fin-2;
       offset:=offset+2;
    END IF;
    pos:=pos+1;
  END LOOP;

  -- 3. Buscamos disyunciones: OR
  pos:=inicio+1; -- En la primera posicin no puede haber un OR (es op. binario)
  WHILE pos<=fin LOOP
    IF CDEG(pos)='OR' THEN
       pos:=pos-1;
       IF F_OR='GREATEST' THEN
          IF CDEG(pos)='1' OR CDEG(pos+2)='1' THEN
             -- El mximo de 1 y 'algo' es: 1
             CDEG(pos):='1'; -- Ese/esos 1/s pueden provenir de condiciones crisp que suponemos TRUE!!
          ELSIF CDEG(pos)='0' OR CDEG(pos+2)='0' THEN
             -- El mximo de 0 y 'algo' es: 'algo'
             IF CDEG(pos)='0' THEN CDEG(pos):=CDEG(pos+2);
             END IF;
          ELSE
             CDEG(pos):='GREATEST('||CDEG(pos)||','||CDEG(pos+2)||')';
          END IF;
       ELSE -- Se est usando otra funcin para evaluar el OR
          CDEG(pos):=F_OR||'('||CDEG(pos)||','||CDEG(pos+2)||')';
       END IF;
       FOR i IN pos+1..(fin-2) LOOP -- Desplazo el resto hacia abajo dos posiciones:
         CDEG(i):=CDEG(i+2);
       END LOOP;
       fin:=fin-2;
       offset:=offset+2;
    END IF;
    pos:=pos+1;
  END LOOP;

  -- A partir de la posicion fin+1, desplazo todo hacia abajo offset posiciones
  max_CDEG:=max_CDEG-offset;
  FOR i IN (fin+1)..max_CDEG LOOP
    CDEG(i):=CDEG(i+offset);
  END LOOP;
END cambia_oper_logicos;

----------------------------------------------------------------------------------------------
-- Cambia operadores lgicos NOT, AND y OR de la tabla CDEG, entre las posiciones inicio y fin:
-- cuando se trata de un CDEG(columna) y puede haber condiciones que se deben eliminar.
-- Si un operando es igual a '\\' indica que se tratar como si no estuviera.
-- Utiliza las funciones calculadas por Lee_OPTIONS;
-- Requisito: Entre las posiciones inicio+1 y fin-1 no deben existir parntesis
----------------------------------------------------------------------------------------------
PROCEDURE cambia_oper_logicos2(CDEG     IN OUT TABLA_CDEG,
                               max_CDEG IN OUT INTEGER,
                               inicio   IN     FSQL_QUERY.indice%TYPE,
                               fin      IN OUT FSQL_QUERY.indice%TYPE) IS
  offset INTEGER:=0; -- Desplazamiento de los valores de la tabla (a partir de fin)
  pos    FSQL_QUERY.posicion%TYPE:=inicio;
BEGIN
  -- Antes de cambiar operadores lgicos vemos si hay parntesis en la primera y ltima posiciones
  IF CDEG(inicio) IN ('(','{') THEN
     fin:=fin-2;
     FOR i IN inicio..fin LOOP
       CDEG(i):=CDEG(i+1);
     END LOOP;
     offset:=2; -- Todo se desplazar 2 posiciones (las de los 2 parntesis)
  END IF;

  -- Cambiamos OPERADORES LGICOS en 3 pasos:
  -- 1. Buscamos negaciones: NOT
  WHILE pos<=fin LOOP
    IF CDEG(pos)='NOT' THEN
       IF CDEG(pos+1)='\\' THEN
          offset:=offset+1;
          fin:=fin-1;
          FOR i IN pos..fin LOOP -- Quito el NOT (y se desplaza todo 1 posicin)
            CDEG(i):=CDEG(i+1);
          END LOOP;
--  dbms_output.put_line('*A*'||fin);
--  FOR i IN inicio..fin LOOP dbms_output.put_line(CDEG(i)); END LOOP;
          -- Buscamos el operador lgico anterior y lo quitamos junto con el \\.
          -- Si este no existe, quitamos el siguiente junto con el \\.
          -- Si tampoco existe dejamos el \\
          IF pos-1>=inicio AND CDEG(pos-1) IN ('AND','OR') THEN
             offset:=offset+2;
             fin:=fin-2;
             FOR i IN (pos-1)..fin LOOP -- Quito el op. anterior y el \\
               CDEG(i):=CDEG(i+2);
             END LOOP;
          ELSIF pos+1<=fin AND CDEG(pos+1) IN ('AND','OR') THEN
             offset:=offset+2;
             fin:=fin-2;
             FOR i IN pos..fin LOOP -- Quito el \\ y el op. siguiente
               CDEG(i):=CDEG(i+2);
             END LOOP;
          END IF;
       ELSE
         IF F_NOT='1-' THEN -- Valor por defecto para la funcin del NOT
            IF CDEG(pos+1)='1' THEN
               CDEG(pos):='0';
            ELSIF CDEG(pos+1)='0' THEN -- Caso: NOT (NOT algo_que_es_1)
               CDEG(pos):='1';
            ELSE
               CDEG(pos):='(1-('||CDEG(pos+1)||'))';
            END IF;
         ELSE -- Se est usando otra funcin para evaluar el NOT
            CDEG(pos):=F_NOT||'('||CDEG(pos+1)||'))';
         END IF;
         fin:=fin-1;
         offset:=offset+1;
         FOR i IN (pos+1)..fin LOOP -- Desplazo el resto hacia abajo una posicin:
           CDEG(i):=CDEG(i+1);
         END LOOP;
       END IF;
    END IF;
    pos:=pos+1;
  END LOOP;

  -- 2. Buscamos conjunciones: AND
  pos:=inicio+1; -- En la primera posicin no puede haber un AND (es op. binario)
  WHILE pos<=fin LOOP
    IF CDEG(pos)='AND' THEN
       pos:=pos-1;
       IF CDEG(pos)='\\' THEN -- Si algn operando es intratable, nos quedamos con el otro
             CDEG(pos):=CDEG(pos+2);
       ELSIF CDEG(pos+2)='\\' THEN
             NULL;
       ELSE
         IF F_AND='LEAST' THEN -- Para evaluar el AND se usa la funcin por defecto: LEAST
            IF CDEG(pos+2)='1' OR CDEG(pos)='0' THEN
               -- El mnimo de 1 y 'algo' es: 'algo'
               -- El mnimo de 0 y 'algo' es: 0
               NULL;
            ELSIF CDEG(pos+2)='0' THEN
               CDEG(pos):='0';
            ELSIF CDEG(pos)='1' THEN
               CDEG(pos):=CDEG(pos+2);
            ELSE  CDEG(pos):='LEAST('||CDEG(pos)||','||CDEG(pos+2)||')';
            END IF;
         ELSE -- Se est usando otra funcin para evaluar el AND
           CDEG(pos):=F_AND||'('||CDEG(pos)||','||CDEG(pos+2)||')';
         END IF;
       END IF;
       FOR i IN pos+1..(fin-2) LOOP -- Desplazo el resto hacia abajo dos posiciones:
         CDEG(i):=CDEG(i+2);
       END LOOP;
       fin:=fin-2;
       offset:=offset+2;
    END IF;
    pos:=pos+1;
  END LOOP;

  -- 3. Buscamos disyunciones: OR
  pos:=inicio+1; -- En la primera posicin no puede haber un OR (es op. binario)
  WHILE pos<=fin LOOP
    IF CDEG(pos)='OR' THEN
       pos:=pos-1;
       IF CDEG(pos)='\\' THEN -- Si algn operando es intratable, nos quedamos con el otro
             CDEG(pos):=CDEG(pos+2);
       ELSIF CDEG(pos+2)='\\' THEN
             NULL;
       ELSE
         IF F_OR='GREATEST' THEN -- Para evaluar el OR se usa la funcin por defecto: GREATEST
            IF CDEG(pos)='1' OR CDEG(pos+2)='1' THEN
                CDEG(pos):='1'; -- El mximo de 1 y 'algo' es: 1
            ELSIF CDEG(pos+2)='0' THEN
                NULL; -- El mximo 'algo' y 0 es: 'algo'
            ELSIF CDEG(pos)='0' THEN
                CDEG(pos):=CDEG(pos+2); -- El mximo de 0 y 'algo' es: 'algo'
            ELSE CDEG(pos):='GREATEST('||CDEG(pos)||','||CDEG(pos+2)||')';
            END IF;
         ELSE -- Se est usando otra funcin para evaluar el OR
           CDEG(pos):=F_OR||'('||CDEG(pos)||','||CDEG(pos+2)||')';
         END IF;
       END IF;
       FOR i IN pos+1..(fin-2) LOOP -- Desplazo el resto hacia abajo dos posiciones:
         CDEG(i):=CDEG(i+2);
       END LOOP;
       fin:=fin-2;
       offset:=offset+2;
    END IF;
    pos:=pos+1;
  END LOOP;

  -- A partir de la posicion fin+1, desplazo todo hacia abajo offset posiciones
  max_CDEG:=max_CDEG-offset;
  FOR i IN (fin+1)..max_CDEG LOOP
    CDEG(i):=CDEG(i+offset);
  END LOOP;
END cambia_oper_logicos2;

----------------------------------------------------------------------------------------------
-- Devuelve el anterior operador lgico, tras una comparacin crisp
----------------------------------------------------------------------------------------------
FUNCTION operador_anterior(columna IN QUERYTYPE) RETURN VARCHAR2 IS
cont INTEGER:=columna.indice-1;
nomb FSQL_QUERY.nombre%TYPE;
BEGIN
  LOOP
    SELECT nombre INTO nomb FROM FSQL_QUERY WHERE indice=cont;
    EXIT WHEN nomb IN ('OR','AND','WHERE');
    cont:=cont-1;
  END LOOP;
  RETURN nomb;
EXCEPTION
  WHEN NO_DATA_FOUND THEN RETURN 'X';
END operador_anterior;

----------------------------------------------------------------------------------------------
-- Devuelve el siguiente operador lgico, tras una comparacin crisp
----------------------------------------------------------------------------------------------
FUNCTION operador_siguiente(columna IN QUERYTYPE) RETURN VARCHAR2 IS
cont INTEGER:=columna.indice+1;
nomb FSQL_QUERY.nombre%TYPE;
BEGIN
  LOOP
    SELECT nombre INTO nomb FROM FSQL_QUERY WHERE indice=cont;
    EXIT WHEN nomb IN ('AND','OR',';',   -- '}',')',
                       'GROUP','CONNECT','START','UNION','INTERSECT','MINUS','ORDER');
    cont:=cont+1;
  END LOOP;
  RETURN nomb;
EXCEPTION
  WHEN NO_DATA_FOUND THEN RETURN 'X';
END operador_siguiente;

----------------------------------------------------------------------------------------------
-- Trata el caso de un CDEG que no es aplicable, ya que no aparece el campo en la clusula WHERE
----------------------------------------------------------------------------------------------
PROCEDURE CDEGnoAplicable(columna IN QUERYTYPE) IS
  inicio  FSQL_QUERY.indice%TYPE:=columna.indice+2;
  fin     FSQL_QUERY.indice%TYPE;
  nomb    FSQL_QUERY.nombre%TYPE;
  atrib   FSQL_QUERY.atributo%TYPE;
BEGIN
  IF columna.posicion<>0 THEN
     -- Si no es aplicable el CDEG pero el usuario lo ha puesto explicitamente,
     -- ponemos un 1, mejor que dar un error (?)
     --FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
     --  ': Atributo difuso '||SUBSTR(stc,1,LENGTH(stc)-1)||
     --  ' es argumento de CDEG y no aparece en la clusula WHERE.');
     UPDATE FSQL_QUERY SET atributo='1' WHERE indice=columna.indice;

  ELSE
     -- Es un CDEG puesto automticamente por un comodn del tipo % (ya que posicion=0)
     -- Si para un atributo CDEG no es aplicable, no aparecer si se usa el comodn %
     -- Borrar toda esa columna: Borrar todo hasta la siguiente coma (inclusive) o FROM (no inclusive)
     -- En caso de FROM borrar todo hasta la coma anterior (inclusive) o SELECT (no inclusive)
--dbms_output.put_line('@@@ '||columna.indice);
     fin:=columna.indice+1;
     LOOP
       SELECT nombre,atributo INTO nomb,atrib FROM FSQL_QUERY WHERE indice=fin;
       EXIT WHEN nomb IN (', en sl','FROM') AND atrib IN (',','FROM');
       fin:=fin+1;
     END LOOP;
--dbms_output.put_line('@@@ '||nomb);
     IF nomb = ', en sl' THEN
        -- Caso 1: Despues del CDEG hay una coma:
        -- Borrar todo hasta la siguiente coma (inclusive)
--dbms_output.put_line('@@@1 '||fin);
        UPDATE FSQL_QUERY SET atributo=NULL WHERE indice BETWEEN columna.indice AND fin;
     ELSE
        -- Encontrado FROM: Borrar todo hasta la coma anterior (inclusive) o SELECT (no inclusive)
        inicio:=columna.indice-1;
        LOOP
          SELECT nombre,atributo INTO nomb,atrib FROM FSQL_QUERY WHERE indice=inicio;
          EXIT WHEN nomb IN (', en sl','SELECT') AND atrib IN (',','SELECT');
          inicio:=inicio-1;
        END LOOP;
        IF nomb = ', en sl' THEN
           -- Caso 2: Despus del CDEG hay un FROM y antes una coma:
           -- Borrar desde la coma (inclusive) hasta el FROM (no inclusive)
--dbms_output.put_line('@@@2 '||inicio||'-'||fin);
           UPDATE FSQL_QUERY SET atributo=NULL WHERE indice BETWEEN inicio AND fin-1;
        ELSE
           -- Caso 3: Despus del CDEG hay un FROM y antes est el SELECT:
           -- Este caso no se puede dar, ya que supondra que el CDEG est slo y que
           -- ha sido puesto automticamente con un comodn %: Con el comodn tambin
           -- habra puesto la columna del argumento del CDEG: SELECT col,CDEG(col) FROM...
           UPDATE FSQL_QUERY SET atributo=NULL WHERE indice BETWEEN inicio+1 AND fin-1;
        END IF;
     END IF;
  END IF;
END CDEGnoAplicable;

----------------------------------------------------------------------------------------------
-- Calcula el Grado de Compatibilidad, CDEG, de un atributo difuso concreto.
-- Este se calcula segn las condiciones en las que aparezca este atributo
-- en la clusula WHERE:
-- NOT A --> 1-A;   A AND B --> min(A,B);   A OR B --> max(A,B);
----------------------------------------------------------------------------------------------
PROCEDURE calcula_CDEG(columna IN QUERYTYPE,
                       MAXind  IN FSQL_QUERY.indice%TYPE) IS
  inicio  FSQL_QUERY.indice%TYPE:=columna.indice+1; -- Inicio de la condicin
  fin     FSQL_QUERY.indice%TYPE; -- Indice que indica el Fin de la condicin
  nomb    FSQL_QUERY.nombre%TYPE; -- Nombre de los tokens
  condic  QUERYTYPE; -- Condicin en la que interviene la columna buscada
  cont    INTEGER:=0;
  tupla   INTEGER; -- tupla en la que termina una llamada a funcin
  stc     VARCHAR2(92); -- 'scheme.table.column': Nombre completo de la columna
  -- Tabla PL/SQL para formar el clculo del CDEG:
  CDEG     TABLA_CDEG;
  max_CDEG INTEGER; -- Nmero de elementos introducidos en la tabla CDEG
  haya_parentesis BOOLEAN:=TRUE; -- Indica si hay parntesis en la condicin

  scheme  VARCHAR2(30); -- Esquema de una tabla/columna
  tab     VARCHAR2(30); -- Tabla de una columna con formato [schema.]tab.columna
  f_tipo FUZZY_COL_LIST.F_TYPE%TYPE;-- Tipo difuso de una columna en FCL
  OBJ    FUZZY_COL_LIST.OBJ#%TYPE;  -- Identificador de un objeto (tabla)
  COL    FUZZY_COL_LIST.COL#%TYPE;  -- Identificador de una columna de una tabla

BEGIN
  -- Quitamos el 'CDEG' que se aadi al final (el ltimo carcter de stc ser '\'):
  stc:=SUBSTR(columna.nombre,1,LENGTH(columna.nombre)-4);
  -- Cambio el ) por un alias para la columna del CDEG:
  UPDATE FSQL_QUERY SET nombre='CDEG c_alias',atributo='"CDEG('||columna.atributo||')"'
    WHERE indice=columna.indice+1;

  -- Calcular posicin del WHERE del SELECT donde aparece esta columna:
  LOOP
    SELECT nombre INTO nomb FROM FSQL_QUERY WHERE indice=inicio;
    EXIT WHEN (nomb='WHERE' AND cont=0) OR inicio=MAXind;
    IF    nomb='SELECT' THEN cont:=cont+1;
    ELSIF nomb='WHERE'  THEN cont:=cont-1;
    END IF;
    inicio:=inicio+1;
  END LOOP;

  IF nomb<>'WHERE' THEN -- Si no hemos encontrado WHERE: consulta sin condicin
     CDEGnoAplicable(columna);
     RETURN;
  END IF;

  -- Posicin donde comienza la condicin:
  inicio:=inicio+1;

  -- Posicin donde termina la condicin:
  -- Seguidores(clausula_where)={ <el fin>,GROUP,CONNECT,START,
  --                              UNION,INTERSECT,MINUS,ORDER,;,) }
  fin:=inicio; cont:=0;
  LOOP
    SELECT nombre INTO nomb FROM FSQL_QUERY WHERE indice=fin;
    EXIT WHEN fin>=MAXind OR
              nomb IN ('GROUP','CONNECT','START','UNION','INTERSECT','MINUS','ORDER',';') OR
              (nomb=')' AND cont=0);
    IF    nomb='(' THEN cont:=cont+1;
    ELSIF nomb=')' THEN cont:=cont-1;
    END IF;
    fin:=fin+1;
  END LOOP;
  IF nomb IN ('GROUP','CONNECT','START','UNION','INTERSECT','MINUS','ORDER',';') THEN
     fin:=fin-1;
  END IF;

  -- Copiamos la condicin del WHERE en la tabla CDEG:
  max_CDEG:=0; cont:=inicio;
  IF stc='*\' THEN -- Se trata de un CDEG(*): Copiarlo todo!    ***** CDEG(*) *****

     WHILE cont<=fin LOOP
--dbms_output.put_line('indices: '||cont||':'||inicio||':'||fin);
       SELECT indice,posicion,nombre,atributo INTO condic FROM FSQL_QUERY
         WHERE indice=cont;
       IF condic.atributo IS NOT NULL THEN
          IF condic.nombre LIKE '%\' THEN -- Si es una columna: condic.atributo LIKE 'FSQL_FUNCTIONS%'
             max_CDEG:=max_CDEG+1;
             CDEG(max_CDEG):=Grado_Condicion(condic,tupla);
             cont:=cont+tupla;

          ELSIF condic.atributo IN ('AND','OR') THEN
             -- Si hay condiciones crisp SIN columnas, como ...AND 4<5 OR..., en CDEG copiaria: AND OR
             IF CDEG(max_CDEG) NOT IN ('NOT','AND','OR') THEN
                max_CDEG:=max_CDEG+1;
             END IF;
             CDEG(max_CDEG):=condic.atributo;
          ELSIF condic.atributo='NOT' THEN
             max_CDEG:=max_CDEG+1;
             CDEG(max_CDEG):=condic.atributo;

          ELSIF condic.atributo IN ('(',')') AND condic.nombre NOT LIKE '%function' THEN
             -- Para no introducir los parntesis de las funciones. Ej: sqrt(...)
             max_CDEG:=max_CDEG+1;
             CDEG(max_CDEG):=condic.atributo;
          ELSIF condic.atributo IN ('{','}') THEN
             max_CDEG:=max_CDEG+1;
             CDEG(max_CDEG):=condic.atributo;

          ELSIF condic.nombre='crisp_col' OR      -- Es una columna crisp o
                condic.nombre LIKE '%column' THEN -- Es columna tipo 1
             -- Las condiciones crisp operadas con OR toman el valor 0 si son FALSAS y 1 si son VERDADERAS
             -- Para hallar ese valor (0  1) se produce la llamada a una funcin:
             max_CDEG:=max_CDEG+1;
             CDEG(max_CDEG):=Grado_CondCrisp(condic,tupla); -- Es realmente una condicin crisp o es una expresin?
             IF CDEG(max_CDEG) IS NULL THEN
                -- NO es una condicin crisp, por lo que nos saltamos este item
                max_CDEG:=max_CDEG-1;
             ELSE
                cont:=cont+tupla;
                IF operador_anterior(condic)<>'OR' AND
                   operador_siguiente(condic)<>'OR' THEN -- Condicin crisp sin un OR:
                   -- Si no interviene en un OR, ponemos 1 para aumentar la eficiencia:
                   -- Ej: WHERE a>b AND Cond
                   --     Para las tuplas en las que a>b   : Min(1,Grado(Cond))
                   --     Para las tuplas en las que NO a>b:
                   --          La condicin NO se cumple, por lo que es intil calcular Min(0,Grado(Cond))
                   --          ya que no aparecern esas tuplas en el resultado.
                   CDEG(max_CDEG):='1';
                END IF;
             END IF;
          END IF;
       END IF;
       cont:=cont+1;
     END LOOP;

  ELSE -- CDEG(columna): Copiar slo donde intervenga dicha columna ***** CDEG(columna) *****
     WHILE cont<=fin LOOP
       SELECT indice,posicion,nombre,atributo INTO condic FROM FSQL_QUERY
         WHERE indice=cont;
--dbms_output.put_line('otro '||condic.indice||':'||condic.nombre||':'||condic.atributo||'==>'||max_CDEG);

       IF condic.nombre NOT LIKE '%\2\' THEN -- Si NO es la columna 2 de una comparacin...
--dbms_output.put_line('&'||substr(condic.atributo,1,10)||','||condic.nombre);
          IF condic.nombre=stc THEN -- Si es la columna que buscamos
             max_CDEG:=max_CDEG+1;
             CDEG(max_CDEG):=Grado_Condicion(condic,tupla);
--dbms_output.put_line('>>>'||substr(CDEG(max_CDEG),1,30)||'---'||tupla);
             cont:=cont+tupla;

          ELSIF condic.nombre LIKE '%column' THEN -- Es columna tipo 1 en condicion crisp
            max_CDEG:=max_CDEG+1;
            IF es_fuzzy_column(condic,scheme,tab,f_tipo,OBJ,COL) AND
               stc=scheme||'.'||tab||'.'||condic.atributo||'\' THEN
--dbms_output.put_line(scheme||'.'||tab||'.'||substr(condic.atributo,1,10)||'\'||' y '||stc);
               IF operador_anterior(condic)='OR' OR
                  operador_siguiente(condic)='OR' THEN -- Condicin crisp con un OR:
                  CDEG(max_CDEG):=Grado_CondCrisp(condic,tupla);
                  IF CDEG(max_CDEG) IS NULL THEN
                     -- NO es una condicin crisp, por lo que nos saltamos este item
                     max_CDEG:=max_CDEG-1;
                  ELSE
                     cont:=cont+tupla;
                  END IF;
               ELSE
                  CDEG(max_CDEG):='1';
               END IF;
            ELSE
               CDEG(max_CDEG):='\\'; -- La \\ indica que se debe eliminar
            END IF;

          ELSIF condic.nombre LIKE '%\' OR condic.nombre='crisp_col' THEN -- Si es otra columna
             max_CDEG:=max_CDEG+1;
             CDEG(max_CDEG):='\\';

          ELSIF condic.atributo IN ('AND','OR') THEN
             -- Si hay condiciones crisp SIN columnas, como ...AND 4<5 OR..., en CDEG copiaria: AND OR
             IF CDEG(max_CDEG) NOT IN ('NOT','AND','OR') THEN
                max_CDEG:=max_CDEG+1;
             END IF;
             CDEG(max_CDEG):=condic.atributo;
          ELSIF condic.atributo='NOT' THEN
             max_CDEG:=max_CDEG+1;
             CDEG(max_CDEG):=condic.atributo;

          ELSIF condic.atributo IN ('{','}') THEN
             max_CDEG:=max_CDEG+1;
             CDEG(max_CDEG):=condic.atributo;
          ELSIF condic.atributo IN ('(',')') AND condic.nombre NOT LIKE '%function' THEN
             -- Para no introducir los parntesis de las funciones. Ej: sqrt(...)
             max_CDEG:=max_CDEG+1;
             CDEG(max_CDEG):=condic.atributo;
          END IF;

       ELSE -- Es la columna 2 de una condicin: col1 <fcomp> col2
          IF condic.nombre=stc||'2\' THEN -- Si, adems, esa columna es la buscada...
             -- Busco hacia atrs a la columna 1: col1
             tupla:=condic.indice-2;
             LOOP
               SELECT indice,posicion,nombre,atributo INTO condic FROM FSQL_QUERY
                 WHERE indice=tupla;
               EXIT WHEN condic.nombre LIKE '%\';
               tupla:=tupla-1;
             END LOOP;
             -- Encontrada col1: Aadirla (como si fuera la col2) a la tabla CDEG en la misma posicin
             -- que el anterior (el anterior debe ser el \\ de la columna1 de esa misma comparacin)
             CDEG(max_CDEG):=Grado_Condicion(condic,tupla);
             -- Aqu no: cont:=cont+tupla; (porque es a partir de la anterior columna)
          END IF;
       END IF;
       cont:=cont+1;
--dbms_output.put_line('1///'||max_CDEG);
--FOR i IN 1..max_CDEG LOOP --dbms_output.put_line(length(CDEG(i)));
--  dbms_output.put_line(substr(CDEG(i),1,20));
--END LOOP;dbms_output.put_line(' ///');
     END LOOP;
  END IF;

  IF max_CDEG=0 THEN -- Si no hay datos introducidos en CDEG
     max_CDEG:=1;    -- introduzco un dato que indica que no es aplicable
     CDEG(1):='\\';
  END IF;

  -- Si la condicin termina en una condicin crisp SIN columnas (como ...OR 3<5),
  -- al final de la tabla CDEG queda un operador (OR en este ejemplo) suelto: Lo obviamos.
  WHILE CDEG(max_CDEG) IN ('NOT','AND','OR') LOOP
    max_CDEG:=max_CDEG-1;
  END LOOP;

--  dbms_output.put_line('1***'||max_CDEG);
--  FOR i IN 1..max_CDEG LOOP --dbms_output.put_line(length(CDEG(i)));
--    dbms_output.put_line(substr(CDEG(i),1,20));
--  END LOOP;

  -- Mientras haya parntesis, cambiar los operadores lgicos dentro de
  -- los parntesis y quitarlos:
  WHILE haya_parentesis LOOP
    inicio:=1; fin:=1;
    -- Buscamos parntesis ms internos: En posiciones inicio y fin respectivamente
    WHILE fin<=max_CDEG AND CDEG(fin) NOT IN (')','}') LOOP
      IF CDEG(fin) IN ('(','{') THEN
         inicio:=fin;
      END IF;
      fin:=fin+1;
    END LOOP;


    IF fin>max_CDEG THEN -- Si no hay ms parntesis, resolver el ltimo nivel y salir
       IF stc='*\' THEN -- Si es un CDEG(*)
            cambia_oper_logicos(CDEG,max_CDEG,1,max_CDEG);
       ELSE cambia_oper_logicos2(CDEG,max_CDEG,1,max_CDEG);
       END IF;
       haya_parentesis:=FALSE;
    ELSE -- Si hay parntesis, quitarlos y resolverlos
--     dbms_output.put_line('cambia: '||inicio||'-'||fin);
       IF stc='*\' THEN -- Si es un CDEG(*)
            cambia_oper_logicos(CDEG,max_CDEG,inicio,fin);
       ELSE cambia_oper_logicos2(CDEG,max_CDEG,inicio,fin);
       END IF;
    END IF;
  END LOOP;

  IF CDEG(1)='\\' THEN -- CDEG no aplicable, ya que no aparece en la clusula WHERE
     CDEGnoAplicable(columna);
  ELSE
     UPDATE FSQL_QUERY SET atributo=CDEG(1) WHERE indice=columna.indice;
  END IF;
END calcula_CDEG;

-- ***** ***** ***** *****  Control de COMODINES (* y %)  ***** ***** ***** *****

----------------------------------------------------------------------------------------------
-- Incluye las columnas cuando se usan comodines de los tipos: "t.*" o "s.t.*"
-- Si las tablas no tienen atributos difusos (tipos 2  3) se dejan los comodines
----------------------------------------------------------------------------------------------
PROCEDURE Incluye_columnas_st_asterisco (columna IN QUERYTYPE) IS
  token_ant FSQL_QUERY.nombre%TYPE; -- Token anterior a la especificacin de la columna
  token_sig FSQL_QUERY.nombre%TYPE; -- Token siguiente
  inicio_FROM FSQL_QUERY.indice%TYPE; -- Posicin donde comienza el FROM
  fin_FROM    FSQL_QUERY.indice%TYPE; -- Posicin donde termina el FROM
  sch    VARCHAR2(30); -- Esquema del comodn
  tab    QUERYTYPE;    -- Tabla del comodn
  st     VARCHAR2(62); -- Esquema.tabla. o tabla. del comodn (segn proceda)
  tsch   VARCHAR2(30); -- Esquema de una tabla del FROM (la tabla es tabla.atributo)
  n_objs INTEGER;      -- Nmero de objetos en una tabla
  Existe BOOLEAN:=FALSE; -- Si existe el s.t del comodn en la clusula FROM
  OBJ    FUZZY_COL_LIST.OBJ#%TYPE;
  COL    FUZZY_COL_LIST.COL#%TYPE;
  OBJ2   FUZZY_COL_LIST.OBJ#%TYPE; -- OBJ,COL del atributo tipo 3 compatible con el tratado
  COL2   FUZZY_COL_LIST.COL#%TYPE;
  Todas  VARCHAR2(2000):=''; -- Donde se acumulan Todas las columnas
  num_columnas INTEGER; -- Nmero de columnas de la tabla con comodines
  nombre_col   DBA_TAB_COLUMNS.COLUMN_NAME%TYPE; -- Nombre de una columna
  f_tipo       FUZZY_COL_LIST.F_TYPE%TYPE;       -- Tipo difuso de una columna
  length3      FUZZY_COL_LIST.LEN%TYPE;          -- Longitud para los tipo 3
BEGIN
--LOOP dbms_output.put_line('->'||length(Todas));
--     Todas:=Todas||'X'||'y';
--     EXIT WHEN length(Todas)=1006;
--END LOOP;
--dbms_output.put_line('FIN: '||length(Todas));
--dbms_output.put_line(Todas);

  -- Calculamos el token anterior a la especificacin de comodn (s.t.c)
  -- dbms_output.put_line('ic:'||columna.posicion);
  IF columna.nombre='t.*' THEN
    SELECT indice,posicion,nombre,atributo INTO tab FROM FSQL_QUERY
      WHERE indice=columna.indice-2;
    st:=tab.atributo||'.';
    sch:=Propietario_Tabla(tab);
    IF sch IS NULL THEN RETURN; END IF; -- La consulta contiene un error en esa tabla
    SELECT nombre INTO token_ant FROM FSQL_QUERY
      WHERE indice=columna.indice-3;
  ELSE -- columna.nombre='s.t.*'
    SELECT indice,posicion,nombre,atributo INTO tab FROM FSQL_QUERY
      WHERE indice=columna.indice-2;
    SELECT atributo INTO sch FROM FSQL_QUERY
      WHERE indice=columna.indice-4;
    st:=sch||'.'||tab.atributo||'.';
    SELECT nombre INTO token_ant FROM FSQL_QUERY
      WHERE indice=columna.indice-5;
  END IF;

  -- Calculamos el token siguiente a la especificacin de comodn
  BEGIN
    SELECT nombre INTO token_sig FROM FSQL_QUERY
      WHERE indice=columna.indice+1;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN token_sig:=NULL; -- Columna al final de la consulta
  END;

  -- Para ver si estamos en la select_list miramos los tokens que rodean la columna
  IF (token_ant IN ('SELECT',', en sl')) AND
     (token_sig IN ('FROM','c_alias',', en sl')) THEN
     IF token_sig='c_alias' THEN
        FSQL_AUX.inserta_error('ERROR SINTCTICO en posicin '||columna.posicion||
          ': Comodn '||st||'* no debe tener alias.');
     END IF;

     -- Encontrar el FROM
     inicio_FROM:=columna.indice+1;
     LOOP
       SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=inicio_FROM;
       EXIT WHEN token_sig='FROM';
       inicio_FROM:=inicio_FROM+1;
     END LOOP;
     inicio_FROM:=inicio_FROM+1;
     -- Encontrar fin del FROM
     fin_FROM:=inicio_FROM+1;
     LOOP
       BEGIN SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=fin_FROM;
       EXCEPTION WHEN NO_DATA_FOUND THEN token_sig:='fin';
       END;
       EXIT WHEN -- Seguidores(tablas)=(...)
            token_sig IN ('fin',';''WHERE','GROUP','CONNECT','START',
                          'UNION','INTERSECT','MINUS','ORDER',')');
       fin_FROM:=fin_FROM+1;
     END LOOP;
     fin_FROM:=fin_FROM-1; -- dbms_output.put_line('ic:'||inicio_FROM||','||fin_FROM);

    -- Verificar la existencia de la tabla del comodn en la lista del FROM:
    FOR tabla IN (SELECT indice,nombre,atributo FROM FSQL_QUERY
                  WHERE  nombre LIKE '%tabla' AND
                         indice BETWEEN inicio_FROM AND fin_FROM) LOOP
      IF tabla.nombre='tabla' THEN
         Existe:=TRUE; -- Ya se ha comprobado al hacer la llamada a la funcin Propietario_Tabla
      ELSE -- tabla.nombre='s.tabla' THEN
         SELECT atributo INTO tsch FROM FSQL_QUERY
           WHERE indice=tabla.indice-2;
      END IF;
      IF tabla.atributo=tab.atributo AND tsch=sch THEN
         -- El s.t del comodn est en la clusula FROM
         Existe:=TRUE;
      END IF;
    END LOOP;

    IF NOT Existe THEN
       FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
         ': No existe la tabla del comodn ('||sch||'.'||tab.atributo||
         '.*) en la clusula FROM correspondiente.');
       RETURN;
    END IF;

    -- Hallamos el OBJ de la tabla
    OBJ:=OBJ_ID(sch,tab.atributo);

    -- Calculamos si hay columnas difusas tipos 2  3: Si no hay: No hacer nada!
    SELECT COUNT(*) INTO n_objs FROM FUZZY_COL_LIST
      WHERE OBJ#=OBJ AND F_TYPE IN (2,3);
    IF n_objs=0 THEN RETURN;
    END IF;

    -- Tenemos un comodn * en una tabla con al menos un atributo tipo 2  3
    -- Borramos el s.t.
--dbms_output.put_line('%%%%%%%:'||columna.nombre||','||columna.indice);
    IF columna.nombre='t.*' THEN
       UPDATE FSQL_QUERY SET atributo=NULL
         WHERE indice BETWEEN columna.indice-2 AND columna.indice-1;
    ELSIF columna.nombre='s.t.*' THEN
       UPDATE FSQL_QUERY SET atributo=NULL
         WHERE indice BETWEEN columna.indice-4 AND columna.indice-1;
    END IF;

    -- Columna por columna de la tabla la incluimos en la lista.
    -- Si el atributo es difuso (tipo 2  3) se incluye la llamada a la funcin que
    -- representa su valor. Suponemos que los atributos difusos tipo 2  3 tienen en
    -- orden sus columnas: primero el tipo (terminado en T)...
--    SELECT COUNT(*) INTO num_columnas FROM DBA_TAB_COLUMNS
--      WHERE OWNER     =Nombre_Owner(sch,tab.atributo) AND
--            TABLE_NAME=Nombre_Table(tab.atributo);
    num_columnas:=Numero_Columnas(sch,tab.atributo);
--dbms_output.put_line('+nc:'||num_columnas);
    COL:=1; -- El atributo COLUMN_ID de DBA_TAB_COLUMNS comienza en 1
    WHILE COL<=num_columnas LOOP
--      SELECT COLUMN_NAME INTO nombre_col FROM DBA_TAB_COLUMNS
--        WHERE OWNER=sch AND TABLE_NAME=tab.atributo AND COLUMN_ID=COL;
      nombre_col:=Nombre_Columna(sch,tab.atributo,COL);
--dbms_output.put_line('ic,nombre:'||nombre_col||'('||OBJ||','||COL||')');
      -- Ver si es difusa:
      BEGIN SELECT F_TYPE,LEN INTO f_tipo,length3 FROM FUZZY_COL_LIST
              WHERE OBJ#=OBJ AND COL#=COL;
      EXCEPTION WHEN NO_DATA_FOUND THEN f_tipo:=0;
      END;
--dbms_output.put_line('ic,ft:'||f_tipo);
      IF COL<>1 THEN Todas:=Todas||','; END IF; -- Concatenamos una coma
      IF f_tipo=2 THEN
         -- La antepongo el "esquema.tabla." y le quito la 'T' final:
         nombre_col:=SUBSTR(nombre_col,1,LENGTH(nombre_col)-1);
         -- Se hace la llamada a la funcin que muestra los atributos difusos
         -- tipo 2 y se le aade al final el nombre de la columna como alias:
         Todas:=Todas||'FSQL_FUNCTIONS.FSHOW2('||OBJ||','||COL||','||
                st||nombre_col||'T,'||st||nombre_col||'1,'||st||nombre_col||'2,'||
                st||nombre_col||'3,'||st||nombre_col||'4)'||nombre_col;
         -- Nos saltamos las columnas que forman parte del atributo difuso tipo 2:
         COL:=COL+5;
      ELSIF f_tipo=3 THEN -- Atributo tipo 3 con longitud length3
         nombre_col:=SUBSTR(nombre_col,1,LENGTH(nombre_col)-1);
         -- Hallamos la columna (OBJ2,COL2) con las etiquetas definidas en FUZZY_NEARNESS_DEF
         BEGIN SELECT OBJ#2,COL#2 INTO OBJ2,COL2 FROM FUZZY_COMPATIBLE_COL
                 WHERE OBJ#1=OBJ AND COL#1=COL;
         EXCEPTION WHEN NO_DATA_FOUND THEN
                 OBJ2:=OBJ; COL2:=COL;
         END;
         IF length3=1 THEN -- Para simplificar la llamada a la funcin que representa los tipo 3
           Todas:=Todas||'FSQL_FUNCTIONS.FSHOW3_LEN1('||OBJ2||','||COL2||','||
                  st||nombre_col||'T,'||st||nombre_col||'P1,'||st||nombre_col||'1)'||nombre_col;
           COL:=COL+3;
         ELSE
           Todas:=Todas||'FSQL_FUNCTIONS.FSHOW3('||OBJ2||','||COL2||','||length3
                  ||','||FSQL_AUX.resto_llamada_tipo3(st||nombre_col,length3)||nombre_col;
           COL:=COL+1+2*length3;
         END IF;
      ELSE -- Crisp
         Todas:=Todas||st||nombre_col;
         COL:=COL+1;
      END IF;
    END LOOP;

    -- Actualizamos con la lista de Todas las columnas de esa tabla:
    UPDATE FSQL_QUERY SET atributo=Todas WHERE indice=columna.indice;

  ELSE
     FSQL_AUX.inserta_error('ERROR SINTCTICO en posicin '||columna.posicion||
       ': En esta posicin no se admiten comodines como "'||st||'*".');
  END IF;
END Incluye_columnas_st_asterisco;

----------------------------------------------------------------------------------------------
-- Incluye las columnas cuando se usa un comodn *
-- Si una tabla no tiene atributos conflictivos (difusos tipos 2  3) se deja como [s.]t.*
----------------------------------------------------------------------------------------------
PROCEDURE Incluye_columnas_asterisco (columna IN QUERYTYPE) IS
  token_sig   FSQL_QUERY.nombre%TYPE; -- Token siguiente
  inicio_FROM FSQL_QUERY.indice%TYPE; -- Posicin donde comienza el FROM
  fin_FROM    FSQL_QUERY.indice%TYPE; -- Posicin donde termina el FROM
  st     VARCHAR2(62); -- Esquema.tabla. o tabla. de una tabla del FROM
  tsch   VARCHAR2(30); -- Esquema de una tabla del FROM (la tabla es tabla.atributo)
  talias VARCHAR2(30); -- Si una tabla tiene alias, usaremos ese alias
  n_objs INTEGER;      -- Nmero de objetos en una tabla
  OBJ    FUZZY_COL_LIST.OBJ#%TYPE;
  COL    FUZZY_COL_LIST.COL#%TYPE;
  OBJ2   FUZZY_COL_LIST.OBJ#%TYPE; -- OBJ,COL del atributo tipo 3 compatible con el tratado
  COL2   FUZZY_COL_LIST.COL#%TYPE;
  Todas  VARCHAR2(2000):='';--FSQL_QUERY.atributo%TYPE:=''; -- Donde se acumulan Todas las columnas
  num_columnas INTEGER; -- Nmero de columnas de la tabla con comodines
  nombre_col   DBA_TAB_COLUMNS.COLUMN_NAME%TYPE; -- Nombre de una columna
  f_tipo       FUZZY_COL_LIST.F_TYPE%TYPE;       -- Tipo difuso de una columna
  length3      FUZZY_COL_LIST.LEN%TYPE;          -- Longitud para los tipo 3
BEGIN
  inicio_FROM:=columna.indice+1;
  -- Encontrar fin del FROM
  fin_FROM:=inicio_FROM+1;
  LOOP
    BEGIN SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=fin_FROM;
    EXCEPTION WHEN NO_DATA_FOUND THEN token_sig:='fin';
    END;
    EXIT WHEN -- Seguidores(tablas)=(...)
         token_sig IN ('fin',';''WHERE','GROUP','CONNECT','START',
                       'UNION','INTERSECT','MINUS','ORDER',')');
    fin_FROM:=fin_FROM+1;
  END LOOP;
  fin_FROM:=fin_FROM-1;

  -- Para cada tabla en la lista del FROM:
  FOR tabla IN (SELECT indice,posicion,nombre,atributo FROM FSQL_QUERY
                WHERE  nombre LIKE '%tabla' AND
                       indice BETWEEN inicio_FROM AND fin_FROM) LOOP

    -- Ver si la tabla usa alias
    BEGIN SELECT nombre INTO talias FROM FSQL_QUERY WHERE indice=tabla.indice+1;
    EXCEPTION WHEN NO_DATA_FOUND THEN talias:=NULL; -- Si la tabla es el ltimo token
    END;
    IF talias='t_alias' THEN -- Tiene alias
       SELECT atributo INTO tabla.atributo FROM FSQL_QUERY WHERE indice=tabla.indice+1;
    END IF;

    -- Hallar esquema de la tabla (tsch):
    IF tabla.nombre='tabla' THEN
         tsch:=Propietario_Tabla(tabla);
         IF tsch IS NULL THEN RETURN; END IF; -- Se ha cometido un error en esa tabla
         st:=tabla.atributo||'.';
    ELSE SELECT atributo INTO tsch FROM FSQL_QUERY WHERE indice=tabla.indice-2;
         st:=tsch||'.'||tabla.atributo||'.';
    END IF;

    -- Hallamos el nmero OBJ de la tabla:
    OBJ:=OBJ_ID(tsch,tabla.atributo);

    -- Calculamos si hay columnas difusas tipos 2  3:
    SELECT COUNT(*) INTO n_objs FROM FUZZY_COL_LIST
      WHERE OBJ#=OBJ AND F_TYPE IN (2,3);

    IF n_objs=0 THEN
       -- Si no hay columnas difusas en esa tabla, la incluimos con comodines: [s.]t.*
       Todas:=Todas||st||'*,';
    ELSE
       -- La tabla que tratamos tiene al menos un atributo difuso tipo 2  3:
       -- Columna por columna de la tabla la incluimos en la lista.
       -- Si el atributo es difuso (tipo 2  3) se incluye la llamada a la funcin que
       -- representa su valor. Suponemos que los atributos difusos tipo 2  3 tienen
       -- en orden sus columnas: primero el tipo (terminado en T)...
--       SELECT COUNT(*) INTO num_columnas FROM DBA_TAB_COLUMNS
--         WHERE OWNER     =Nombre_Owner(tsch,tabla.atributo) AND
--               TABLE_NAME=Nombre_Table(tabla.atributo);
       num_columnas:=Numero_Columnas(tsch,tabla.atributo);
--dbms_output.put_line('ic,nc:'||num_columnas);
       -- Para cada columna de esta tabla:
       COL:=1; -- El atributo COLUMN_ID de DBA_TAB_COLUMNS comienza en 1
       WHILE COL<=num_columnas LOOP
--         SELECT COLUMN_NAME INTO nombre_col FROM DBA_TAB_COLUMNS
--           WHERE OWNER=tsch AND TABLE_NAME=tabla.atributo AND COLUMN_ID=COL;
       nombre_col:=Nombre_Columna(tsch,tabla.atributo,COL);
--dbms_output.put_line('ic,nombre:'||nombre_col);
         -- Ver si es difusa:
         BEGIN SELECT F_TYPE,LEN INTO f_tipo,length3 FROM FUZZY_COL_LIST
                 WHERE OBJ#=OBJ AND COL#=COL;
         EXCEPTION WHEN NO_DATA_FOUND THEN f_tipo:=0;
         END;
--dbms_output.put_line('ic,ft:'||f_tipo);
         IF f_tipo=2 THEN
            -- La antepongo el "esquema.tabla." y le quito la 'T' final:
            nombre_col:=SUBSTR(nombre_col,1,LENGTH(nombre_col)-1);
            -- Se hace la llamada a la funcin que muestra los atributos difusos
            -- tipo 2 y se le aade al final el nombre de la columna como alias:
            Todas:=Todas||'FSQL_FUNCTIONS.FSHOW2('||OBJ||','||COL||','||
                   st||nombre_col||'T,'||st||nombre_col||'1,'||st||nombre_col||'2,'||
                   st||nombre_col||'3,'||st||nombre_col||'4)'||nombre_col||',';
            -- Nos saltamos las columnas que forman parte del atributo difuso tipo 2:
            COL:=COL+5;
         ELSIF f_tipo=3 THEN -- Atributo tipo 3 con longitud length3
            nombre_col:=SUBSTR(nombre_col,1,LENGTH(nombre_col)-1);
            -- Hallamos la columna (OBJ2,COL2) con las etiquetas definidas en FUZZY_NEARNESS_DEF
            BEGIN SELECT OBJ#2,COL#2 INTO OBJ2,COL2 FROM FUZZY_COMPATIBLE_COL
                    WHERE OBJ#1=OBJ AND COL#1=COL;
            EXCEPTION WHEN NO_DATA_FOUND THEN
                    OBJ2:=OBJ; COL2:=COL;
            END;
            IF length3=1 THEN -- Para simplificar la llamada para representar los tipo 3
               Todas:=Todas||'FSQL_FUNCTIONS.FSHOW3_LEN1('||OBJ2||','||COL2||','||
                      st||nombre_col||'T,'||st||nombre_col||'P1,'||st||nombre_col||'1)'||nombre_col||',';
               COL:=COL+3;
            ELSE
              Todas:=Todas||'FSQL_FUNCTIONS.FSHOW3('||OBJ2||','||COL2||','||length3
                     ||','||FSQL_AUX.resto_llamada_tipo3(st||nombre_col,length3)||nombre_col||',';
              COL:=COL+1+2*length3;
            END IF;
         ELSE -- Crisp
            Todas:=Todas||st||nombre_col||',';
            COL:=COL+1;
         END IF;

       END LOOP; --WHILE
    END IF;
  END LOOP; --FOR

  -- Actualizamos con la lista de Todas las columnas (quitando la ultima coma):
  --dbms_output.put_line(':'||columna.indice);
  UPDATE FSQL_QUERY SET atributo=SUBSTR(Todas,1,LENGTH(Todas)-1)
    WHERE indice=columna.indice;
END Incluye_columnas_asterisco;

----------------------------------------------------------------------------------------------
-- Incluye las columnas cuando se usan comodines de los tipos: "t.%" o "s.t.%"
-- Si las tablas no tienen atributos conflictivos (difusos tipos 2  3) se cambian
-- por "t.*" o "s.t.*" y se incluyen los CDEG de los tipos 1, si los hay.
-- Los CDEG son incluidos en la tabla FSQL_QUERY como si hubiesen sido escritos en la
-- consulta inicial, poniendo su posicion a cero, y son tratados, como un CDEG normal,
-- por el procedimiento "calcula_CDEG".
-- Si un atributo difuso no aparece en la clusula WHERE, su CDEG no es aplicable y por tanto
-- no aparecer su CDEG si se usa el comodn % (ser eliminado en calcula_CDEG).
----------------------------------------------------------------------------------------------
PROCEDURE Incluye_columnas_st_porciento (columna IN OUT QUERYTYPE,
                                         MAXind  IN OUT FSQL_QUERY.indice%TYPE)
IS
  token_ant FSQL_QUERY.nombre%TYPE; -- Token anterior a la especificacin de la columna
  token_sig FSQL_QUERY.nombre%TYPE; -- Token siguiente
  inicio_FROM FSQL_QUERY.indice%TYPE; -- Posicin donde comienza el FROM
  fin_FROM    FSQL_QUERY.indice%TYPE; -- Posicin donde termina el FROM
  sch    VARCHAR2(30); -- Esquema del comodn
  tab    QUERYTYPE;    -- Tabla del comodn
  st     VARCHAR2(62); -- Esquema.tabla. o tabla. del comodn (segn proceda)
  tsch   VARCHAR2(30); -- Esquema de una tabla del FROM (la tabla es tabla.atributo)
  n_objs INTEGER;      -- Nmero de objetos en una tabla
  Existe BOOLEAN:=FALSE; -- Si existe el s.t del comodn en la clusula FROM
  OBJ    FUZZY_COL_LIST.OBJ#%TYPE;
  nombre_col DBA_TAB_COLUMNS.COLUMN_NAME%TYPE; -- Nombre de una columna
BEGIN
  -- Calculamos el token anterior a la especificacin de comodn (s.t.c)
  IF columna.nombre='t.%' THEN
    SELECT indice,posicion,nombre,atributo INTO tab FROM FSQL_QUERY
      WHERE indice=columna.indice-2;
    st:=tab.atributo||'.';
    sch:=Propietario_Tabla(tab);
    SELECT nombre INTO token_ant FROM FSQL_QUERY WHERE indice=columna.indice-3;
  ELSE --columna.nombre='s.t.%'
    SELECT indice,posicion,nombre,atributo INTO tab FROM FSQL_QUERY WHERE indice=columna.indice-2;
    SELECT atributo INTO sch FROM FSQL_QUERY WHERE indice=columna.indice-4;
    st:=sch||'.'||tab.atributo||'.';
    SELECT nombre INTO token_ant FROM FSQL_QUERY WHERE indice=columna.indice-5;
  END IF;

  -- Calculamos el token siguiente a la especificacin de comodn
  BEGIN
    SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=columna.indice+1;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN token_sig:=NULL; -- Columna al final de la consulta
  END;

  -- Para ver si estamos en la select_list miramos los tokens que rodean la columna
  IF (token_ant IN ('SELECT',', en sl')) AND
     (token_sig IN ('FROM','c_alias',', en sl')) THEN
     IF token_sig='c_alias' THEN
        FSQL_AUX.inserta_error('ERROR SINTCTICO en posicin '||columna.posicion||
          ': Comodn '||st||'% no debe tener alias.');
     END IF;

     -- Encontrar el FROM (para determinar si las tablas con % estan en la clusula FROM)
     inicio_FROM:=columna.indice+1;
     LOOP
       SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=inicio_FROM;
       EXIT WHEN token_sig='FROM';
       inicio_FROM:=inicio_FROM+1;
     END LOOP;
     inicio_FROM:=inicio_FROM+1;
     -- Encontrar fin del FROM
     fin_FROM:=inicio_FROM+1;
     LOOP
       BEGIN SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=fin_FROM;
       EXCEPTION WHEN NO_DATA_FOUND THEN token_sig:='fin';
       END;
       EXIT WHEN -- Seguidores(tablas)=(...)
            token_sig IN ('fin',';''WHERE','GROUP','CONNECT','START',
                          'UNION','INTERSECT','MINUS','ORDER',')');
       fin_FROM:=fin_FROM+1;
     END LOOP;
     fin_FROM:=fin_FROM-1; -- dbms_output.put_line('ic:'||inicio_FROM||','||fin_FROM);

    -- Verificar la existencia de la tabla del comodn en la lista del FROM:
    FOR tabla IN (SELECT indice,nombre,atributo FROM FSQL_QUERY
                  WHERE  nombre LIKE '%tabla' AND
                         indice BETWEEN inicio_FROM AND fin_FROM) LOOP
      IF tabla.nombre='tabla' THEN
         Existe:=TRUE; -- Ya se ha comprobado al hacer la llamada a la funcin Propietario_Tabla
      ELSE -- tabla.nombre='s.tabla' THEN
         SELECT atributo INTO tsch FROM FSQL_QUERY WHERE indice=tabla.indice-2;
      END IF;
      IF tabla.atributo=tab.atributo AND tsch=sch THEN
         -- El s.t del comodn est en la clusula FROM
         Existe:=TRUE;
      END IF;
    END LOOP;

    IF NOT Existe THEN
       FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
         ': No existe la tabla del comodn ('||sch||'.'||tab.atributo||
         '.%) en la clusula FROM correspondiente.');
       RETURN;
    END IF;

    -- Hallamos el OBJ de la tabla
    OBJ:=OBJ_ID(sch,tab.atributo);

    -- Calculamos si hay columnas difusas tipos 2  3: Si no hay: No hacer nada!
    SELECT COUNT(*) INTO n_objs FROM FUZZY_COL_LIST WHERE OBJ#=OBJ AND F_TYPE IN (2,3);
    IF n_objs=0 THEN -- Cambiamos el % por un *
       UPDATE FSQL_QUERY SET atributo='*' WHERE indice=columna.indice;
       -- Calculamos si hay columnas difusas tipos 1:
       SELECT COUNT(*) INTO n_objs FROM FUZZY_COL_LIST WHERE OBJ#=OBJ AND F_TYPE=1;
       IF n_objs=0 THEN
          -- Sin atributos difusos (ni tipo 1, ni 2, ni 3)
          RETURN;
       ELSE
          -- Slo tiene atributos difusos tipo 1: Aadir sus CDEG
          FOR COL_id IN (SELECT COl# FROM FUZZY_COL_LIST WHERE OBJ#=OBJ AND F_TYPE=1) LOOP
--            SELECT COLUMN_NAME INTO nombre_col FROM DBA_TAB_COLUMNS
--              WHERE OWNER=sch AND TABLE_NAME=tab.atributo AND COLUMN_ID=COL_id.COL#;
            nombre_col:=Nombre_Columna(sch,tab.atributo,COL_id.COL#);
            IF columna.nombre='t.%' THEN
               UPDATE FSQL_QUERY SET indice=indice+7
                 WHERE indice BETWEEN columna.indice+1 AND MAXind;
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+1,0,', en sl',',');
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+2,0,'CDEG','CDEG');
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+3,0,'( function','(');
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+4,0,'t.',tab.atributo);
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+5,0,'.','.');
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+6,0,'t.column',nombre_col);
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+7,0,') function',')');
               columna.indice:=columna.indice+7; -- Por si hay ms atributos tipo 1
               MAXind:=MAXind+7;
            ELSE --columna.nombre='s.t.%'
               UPDATE FSQL_QUERY SET indice=indice+9
                 WHERE indice BETWEEN columna.indice+1 AND MAXind;
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+1,0,', en sl',',');
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+2,0,'CDEG','CDEG');
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+3,0,'( function','(');
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+4,0,'s.t.',sch);
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+5,0,'.','.');
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+6,0,'t.',tab.atributo);
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+7,0,'.','.');
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+8,0,'s.t.column',nombre_col);
               INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+9,0,') function',')');
               columna.indice:=columna.indice+9;
               MAXind:=MAXind+9;
            END IF;
          END LOOP;
          RETURN;
       END IF;
    END IF;

    -- Tenemos un comodn % en una tabla con al menos un atributo tipo 2  3
    -- Camiamos el [s.]t.% por un [s.]t.* de nombre y el % por * en el atributo
    columna.nombre:=SUBSTR(columna.nombre,1,LENGTH(columna.nombre)-1)||'*';
    columna.atributo:='*';
    -- Incluimos las columnas de esa tabla:
    Incluye_columnas_st_asterisco(columna);

    -- Incluimos los CDEG pertinentes de los atributos difusos tipos 1, 2 y 3:
    FOR COL IN (SELECT COL#,F_TYPE FROM FUZZY_COL_LIST
                  WHERE OBJ#=OBJ AND F_TYPE BETWEEN 1 AND 3 ORDER BY COL#) LOOP
--dbms_output.put_line('COL---'||COL.COL#);
--      SELECT COLUMN_NAME INTO nombre_col FROM DBA_TAB_COLUMNS
--        WHERE OWNER=sch AND TABLE_NAME=tab.atributo AND COLUMN_ID=COL.COL#;
    nombre_col:=Nombre_Columna(sch,tab.atributo,COL.COL#);
      IF COL.F_TYPE IN (2,3) THEN -- Le quitamos la ltima letra del nombre: la 'T'
         nombre_col:=SUBSTR(nombre_col,1,LENGTH(nombre_col)-1);
      END IF;
--dbms_output.put_line('&&: '||columna.indice||','||columna.nombre||','||columna.atributo);
      IF columna.nombre='t.*' THEN
         UPDATE FSQL_QUERY SET indice=indice+7
           WHERE indice BETWEEN columna.indice+1 AND MAXind;
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+1,0,', en sl',',');
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+2,0,'CDEG','CDEG');
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+3,0,'( function','(');
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+4,0,'t.',tab.atributo);
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+5,0,'.','.');
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+6,0,'t.column',nombre_col);
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+7,0,') function',')');
         columna.indice:=columna.indice+7;
         MAXind:=MAXind+7;
      ELSE --columna.nombre='s.t.*' (Ya se ha cambiado el % por el *)
         UPDATE FSQL_QUERY SET indice=indice+9
           WHERE indice BETWEEN columna.indice+1 AND MAXind;
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+1,0,', en sl',',');
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+2,0,'CDEG','CDEG');
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+3,0,'( function','(');
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+4,0,'s.t.',sch);
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+5,0,'.','.');
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+6,0,'t.',tab.atributo);
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+7,0,'.','.');
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+8,0,'s.t.column',nombre_col);
         INSERT INTO FSQL_QUERY VALUES (AUDSID,columna.indice+9,0,') function',')');
         columna.indice:=columna.indice+9;
         MAXind:=MAXind+9;
      END IF;
    END LOOP;

  ELSE
     FSQL_AUX.inserta_error('ERROR SINTCTICO en posicin '||columna.posicion||
       ': En esta posicin no se admiten comodines como "'||st||'%".');
  END IF;
END Incluye_columnas_st_porciento;

----------------------------------------------------------------------------------------------
-- Incluye las columnas cuando se usa un comodn %
-- Si las tablas no tienen atributos conflictivos (difusos tipos 2  3) se cambian
-- por "t.*" o "s.t.*" y se incluyen los CDEG de los tipos 1, si los hay.
-- Si un atributo difuso no aparece en la clusula WHERE, su CDEG no es aplicable y por tanto
-- no aparecer su CDEG si se usa el comodn %, i.e. ser eliminado en calcula_CDEG.
----------------------------------------------------------------------------------------------
PROCEDURE Incluye_columnas_porciento (columna IN QUERYTYPE,
                                      MAXind  IN OUT FSQL_QUERY.indice%TYPE)
IS
  token_sig   FSQL_QUERY.nombre%TYPE; -- Token siguiente
  talias      FSQL_QUERY.nombre%TYPE; -- Para ver si la tabla tiene alias
  inicio_FROM FSQL_QUERY.indice%TYPE; -- Posicin donde comienza el FROM
  fin_FROM    FSQL_QUERY.indice%TYPE; -- Posicin donde termina el FROM
  ind         FSQL_QUERY.indice%TYPE; -- Indice
  tsch   VARCHAR2(30); -- Esquema de una tabla del FROM (la tabla es tabla.atributo)
BEGIN
  inicio_FROM:=columna.indice+1;
  -- Encontrar fin del FROM
  fin_FROM:=inicio_FROM+1;
  LOOP
    BEGIN SELECT nombre INTO token_sig FROM FSQL_QUERY WHERE indice=fin_FROM;
    EXCEPTION WHEN NO_DATA_FOUND THEN token_sig:='fin';
    END;
    EXIT WHEN -- Seguidores(tablas)=(...)
         token_sig IN ('fin',';''WHERE','GROUP','CONNECT','START',
                       'UNION','INTERSECT','MINUS','ORDER',')');
    fin_FROM:=fin_FROM+1;
  END LOOP;
  fin_FROM:=fin_FROM-1;

  -- Borrar el "%": Para ello copiamos el SELECT de la tupla anterior encima y
  -- borramos el atributo de dicha tupla anterior (esto se hace para que el SELECT
  -- quede justo antes de lo que se inserte a continuacin y as, dara positivo cuando
  -- se compruebe si est en la select-list).
  ind:=columna.indice;
  UPDATE FSQL_QUERY SET (posicion,nombre,atributo)=
         (SELECT posicion,nombre,atributo FROM FSQL_QUERY WHERE indice=ind-1)
         WHERE indice=ind;
  UPDATE FSQL_QUERY SET atributo=NULL WHERE indice=ind-1;

  -- Para cada tabla en la lista del FROM:
  FOR tabla IN (SELECT indice,nombre,atributo FROM FSQL_QUERY
                WHERE  nombre LIKE '%tabla' AND
                       indice BETWEEN inicio_FROM AND fin_FROM) LOOP
    -- Ver si la tabla usa alias
    BEGIN SELECT nombre INTO talias FROM FSQL_QUERY
            WHERE indice=tabla.indice+1+(ind-columna.indice);
    EXCEPTION WHEN NO_DATA_FOUND THEN talias:=NULL;
    END;
    IF talias='t_alias' THEN -- Tiene alias
       SELECT atributo INTO tabla.atributo FROM FSQL_QUERY
         WHERE indice=tabla.indice+1+(ind-columna.indice);
    END IF;

    -- Hallar esquema de la tabla (tsch):
    IF tabla.nombre='tabla' THEN
       UPDATE FSQL_QUERY SET indice=indice+4
         WHERE indice BETWEEN ind+1 AND MAXind;
       INSERT INTO FSQL_QUERY VALUES (AUDSID,ind+1,0,'t.',tabla.atributo);
       INSERT INTO FSQL_QUERY VALUES (AUDSID,ind+2,0,'.','.');
       INSERT INTO FSQL_QUERY VALUES (AUDSID,ind+3,0,'t.%','%');
       INSERT INTO FSQL_QUERY VALUES (AUDSID,ind+4,0,', en sl',',');
       ind:=ind+4;
--dbms_output.put_line('&&: '||tabla.indice||','||tabla.nombre||','||tabla.atributo||','||tsch||','||ind);
       MAXind:=MAXind+4;
    ELSE -- tabla.nombre='s.tabla'
       SELECT atributo INTO tsch FROM FSQL_QUERY
         WHERE indice=tabla.indice-2+(ind-columna.indice);
--dbms_output.put_line('&&: '||tabla.indice||','||tabla.nombre||','||tabla.atributo||','||tsch);
       UPDATE FSQL_QUERY SET indice=indice+6
         WHERE indice BETWEEN ind+1 AND MAXind;
       INSERT INTO FSQL_QUERY VALUES (AUDSID,ind+1,0,'s.t.',tsch);
       INSERT INTO FSQL_QUERY VALUES (AUDSID,ind+2,0,'.','.');
       INSERT INTO FSQL_QUERY VALUES (AUDSID,ind+3,0,'t.',tabla.atributo);
       INSERT INTO FSQL_QUERY VALUES (AUDSID,ind+4,0,'.','.');
       INSERT INTO FSQL_QUERY VALUES (AUDSID,ind+5,0,'s.t.%','%');
       INSERT INTO FSQL_QUERY VALUES (AUDSID,ind+6,0,', en sl',',');
       ind:=ind+6;
       MAXind:=MAXind+6;
    END IF;
  END LOOP; --FOR

  -- Borramos la ltima coma introducida:
  UPDATE FSQL_QUERY SET atributo=NULL WHERE indice=ind;
END Incluye_columnas_porciento;

----------------------------------------------------------------------------------------------
-- ANALIZADOR SEMNTICO para una consulta en FSQL
-- Requisito: La sentencia de FSQL_QUERY debe ser correcta lxica y sintcticamente.
-- Devuelve : Nmero de errores encontrados (0 si ha sido correcta).
-- Modifica en FSQL_QUERY la columna atributo con la traduccin de FSQL a SQL
----------------------------------------------------------------------------------------------
FUNCTION semantico RETURN INTEGER IS
  -- Cursor para las tablas de la consulta (clusula FROM):
  CURSOR tablas IS
     SELECT indice,posicion,nombre,atributo FROM FSQL_QUERY
     WHERE  nombre LIKE '%tabla';
  columna QUERYTYPE;
  ind     FSQL_QUERY.indice%TYPE;
  MAXind  FSQL_QUERY.indice%TYPE;
  nombre  FSQL_QUERY.nombre%TYPE;
  scheme  VARCHAR2(30); -- Esquema de una tabla/columna
  tab     VARCHAR2(30); -- Tabla de una columna con formato [schema.]tab.columna
  stalias VARCHAR2(61); -- Para formar el s.t de un alias de tabla (t_alias)
  tabla   QUERYTYPE;    -- Tablas de la lista de tokens
  n_objs INTEGER;      -- Nmero de objetos en una tabla
  f_tipo FUZZY_COL_LIST.F_TYPE%TYPE;-- Tipo difuso de una columna en FCL
  OBJ    FUZZY_COL_LIST.OBJ#%TYPE;  -- Identificador de un objeto (tabla)
  COL    FUZZY_COL_LIST.COL#%TYPE;  -- Identificador de una columna de una tabla
  hay_NOT BOOLEAN:=FALSE; -- Para detectar si hay dos NOT seguidos
BEGIN
  SELECT USERENV('SESSIONID') INTO AUDSID FROM DUAL;
  SELECT MAX(indice) INTO MAXind FROM FSQL_QUERY;

  ------------------------------------
  -- Capturo los alias de tablas:
  n_objs:=0; -- Nmero de objetos en la tabla de los alias ALL_ALIAS
  FOR alias IN (SELECT atributo,indice FROM FSQL_QUERY
                  WHERE nombre='t_alias') LOOP
    SELECT nombre,atributo INTO tab,stalias FROM FSQL_QUERY
      WHERE indice=alias.indice-1;
    IF tab='s.tabla' THEN -- Si tiene esquema la tabla del alias, lo capturo:
       SELECT atributo INTO scheme FROM FSQL_QUERY
         WHERE indice=alias.indice-3;
    ELSE -- tab='tabla'
       SELECT USER INTO scheme FROM DUAL;
    END IF;

    -- Relleno un elemento de la tabla ALL_ALIAS con los datos de este alias
    n_objs:=n_objs+1;
    ALL_ALIAS(n_objs).N_alias :=alias.atributo;
    ALL_ALIAS(n_objs).N_scheme:=scheme;
    ALL_ALIAS(n_objs).N_table :=stalias;
  END LOOP;

  -----------------------------------------------
  -- Busco las tablas de la consulta (clusula FROM):
  -- Si no se inserta un error es que EXISTE una tabla/vista accesible por el usuario: Perfecto!
  FOR tabla IN tablas LOOP
    -- Verificar la existencia de esa tabla o vista:
    IF tabla.nombre='tabla' THEN
       -- Si no se pone el propietario se supone el usuario (se ignoran los sinnimos pblicos)
       SELECT USER into scheme from DUAL;
    ELSE -- tabla.nombre='s.tabla' THEN
       SELECT atributo INTO scheme FROM FSQL_QUERY WHERE indice=tabla.indice-2;
    END IF;

    -- Ver si esta tabla (o vista...) es accesible por el usuario actual
    SELECT count(*) INTO n_objs FROM ACCESSIBLE_TABLES
      WHERE TABLE_NAME=tabla.atributo AND OWNER=scheme;

    IF n_objs=0 THEN
       FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||tabla.posicion
          ||': Tabla o vista '||tabla.atributo||' no existe.');
    ELSIF n_objs>1 THEN
       FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||tabla.posicion
          ||': Tabla o vista '||tabla.atributo||' ambigua (no completamente especificada).');
    END IF;
  END LOOP;

  -----------------------------------------------
  -- Si hay algn error, no seguimos analizando
  SELECT COUNT(*) INTO n_objs FROM FSQL_ERRORS;
  IF n_objs>0 THEN
     COMMIT;
     RETURN n_objs;
  END IF;

  ------------------------------------
  -- Tratar COMODINES, si existen. En la select_list pueden aparecer los siguientes seis tipos
  -- de comodines: * (TODAS las columnas),
  --               % (TODO: columnas y los CDEG que correspondan),
  --               t.*, t.%, s.t.* y s.t.%.
  ind:=1;
  WHILE ind<=MAXind LOOP
    SELECT indice,posicion,nombre,atributo INTO columna FROM FSQL_QUERY WHERE indice=ind;
--  dbms_output.put_line('-->'||ind||':'||columna.nombre);
    IF columna.nombre='TODAS' THEN
      Incluye_columnas_asterisco(columna);
    ELSIF columna.nombre IN ('t.*','s.t.*') THEN
      Incluye_columnas_st_asterisco(columna);
    ELSIF columna.nombre IN ('t.%','s.t.%') THEN
      Incluye_columnas_st_porciento(columna,MAXind);
      ind:=columna.indice; -- El proced. anterior inserta filas: seguimos al final de lo insertado
    ELSIF columna.nombre='TODO' THEN
      Incluye_columnas_porciento(columna,MAXind);
    END IF;
    ind:=ind+1;
  END LOOP;

  -- Leer las opciones disponibles en la vista FSQL_OPTIONS:
  --   Funciones establecidas para los operadores lgicos (NOT, AND y OR) y
  --   Tratamiento de atributos difusos en posiciones inusuales
  Lee_OPTIONS;

  ------------------------------------
  -- Recorro la consulta buscando las columnas (las no tratadas an, en cada paso):
  FOR ind IN 1..MAXind LOOP
    SELECT indice,posicion,nombre,atributo INTO columna FROM FSQL_QUERY WHERE indice=ind;

    IF columna.nombre LIKE '%column' THEN
       -- Columna no tratada an.
       -- dbms_output.put_line(columna.nombre||' '||columna.atributo);
       IF es_fuzzy_column(columna,scheme,tab,f_tipo,OBJ,COL) THEN
          -- Si la columna es difusa: Tratamiento especial para la seora!
          -- Si es tratada satisfactoriamente se marca en el campo nombre de FSQL_QUERY
          trata_fuzzy_column(columna,scheme,tab,f_tipo,OBJ,COL);
       END IF;
    ELSIF columna.nombre='NOT' THEN -- Para detectar si hay dos NOT seguidos
       IF hay_NOT THEN
          FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
            ': No se pueden poner dos negaciones (NOT) juntas. Use parntesis.');
       ELSE hay_NOT:=TRUE;
       END IF;
    ELSE hay_NOT:=FALSE;
    END IF;
  END LOOP;

  ------------------------------------
  -- Buscando palabras reservadas que deberan haberse tratado:
  -- UNKNOWN, UNDEFINED, fcomparador, CDEG
  FOR ind IN 1..MAXind LOOP
    SELECT indice,posicion,nombre,atributo INTO columna FROM FSQL_QUERY WHERE indice=ind;
    IF columna.atributo IN ('UNKNOWN','UNDEFINED') THEN
       FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
         ': Constante difusa '||columna.atributo||' est mal situada.');
    ELSIF columna.atributo IN ('FGT','FLT','FEQ','FGEQ','FLEQ','MGT','MLT',
                               'NFGT','NFLT','NFEQ','NFGEQ','NFLEQ','NMGT','NMLT') THEN
       FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
         ': Comparador difuso '||columna.atributo||
         ' debe tener una columna difusa justo a su izquierda (tipo 1, 2  3). Formato: <fuzzy_column> '||
         columna.atributo||' <expr>.');

    ELSIF columna.atributo='CDEG' THEN
       FSQL_AUX.inserta_error('ERROR SEMNTICO en posicin '||columna.posicion||
         ': CDEG (Compatibility Degree) slo es aplicable a atributos con tratamiento difuso (tipos 1, 2  3).');
    END IF;
  END LOOP;

  ------------------------------------
  -- Ver si ha habido algn error: En caso afirmativo devolver el nmero de errores encontrados
  -- Si usamos una columna incorrectamente (ej: ... where XXX feq $algo), entonces la funcin
  -- calcula_CDEG casca. Por eso, antes de entrar en ella nos aseguramos que no hay errores:
  SELECT COUNT(*) INTO n_objs FROM FSQL_ERRORS;
  IF n_objs>0 THEN
     COMMIT;
     RETURN n_objs;
  END IF;

  ------------------------------------
  -- Para calcular los CDEG: Busco si hemos encontrado alguna columna como arg. de CDEG
  FOR colCDEG IN (SELECT indice,posicion,nombre,atributo FROM FSQL_QUERY
                    WHERE nombre LIKE '%\CDEG') LOOP
    -- Calculo CDEG(fcolumn) y se sustituye por esta funcin:
    calcula_CDEG(colCDEG,MAXind);
  END LOOP;

  -- Tratamiento de los alias que provoca un CDEG(...):
  -- En un CDEG(...) el parntesis ) es sustituido por lo que ser el alias: "CDEG(...)"
  -- Problemas: 1. Si ya tiene alias, tendremos 2 alias seguidos (Solucin: Quitarlo)
  --            2. Si forma parte de una operacin: Ej: ROUND(CDEG(...),2)
  --               (Solucin: Buscar el token siguiente que indique el fin de esa columna e
  --                insertar el alias justo antes de l)
  FOR aliasCDEG IN (SELECT indice FROM FSQL_QUERY
                      WHERE nombre = 'CDEG c_alias') LOOP
    ind:=aliasCDEG.indice+1;
    LOOP
      BEGIN
        SELECT nombre INTO nombre FROM FSQL_QUERY WHERE indice=ind;
      EXCEPTION WHEN NO_DATA_FOUND THEN nombre:='fin';
      END;
      EXIT WHEN nombre IN -- Fin de columna: alias de columna (o CDEG), coma (,) en la sl, FROM
           ('c_alias','CDEG c_alias',', en sl','FROM','fin');
      ind:=ind+1;
    END LOOP;

    ind:=ind-1;
    IF nombre IN ('c_alias','CDEG c_alias') THEN
       -- Eliminar el alias introducido
       UPDATE FSQL_QUERY SET atributo='' WHERE indice=aliasCDEG.indice;
    ELSIF nombre IN (', en sl','FROM') AND (ind<>aliasCDEG.indice) THEN
       -- Insertar el alias introducido justo antes:
       -- 1: Abrir hueco:
       UPDATE FSQL_QUERY SET indice=-1       WHERE indice = aliasCDEG.indice;
       UPDATE FSQL_QUERY SET indice=indice-1 WHERE indice BETWEEN aliasCDEG.indice+1 AND ind;
       -- 2: Establecerlo en su nuevo lugar:
       --    NO cambiamos el alias a pesar de que no es un CDEG aislado, sino una operacin con l.
       --    Ej: select cdeg(distancia)+1 ...
       --    CONSEJO: Lo mejor es poner alias en las columnas con operaciones de las consultas.
       --             Si no se ponen pueden dar nombres de columna inesperados.
       UPDATE FSQL_QUERY SET indice=ind,nombre='op c_alias' WHERE indice=-1;
    END IF;
  END LOOP;

  COMMIT;
  SELECT COUNT(*) INTO n_objs FROM FSQL_ERRORS;
  RETURN n_objs;
END semantico;

END FSQL_SEMANTIC;
/
